"""
Authors: Peter Mawhorter
Consulted:
Date: 2024-10-30
Purpose: Tests for measureMunger.py
"""

from optimism import (
    freshTestSuite,
    testFunctionMaybe,
    testBlock,
    Loop,
    showOverview
)

import wavesynth

import measureMunger


#--------------#
# Testing data #
#--------------#

runLengths = [3, 4, 3, 2, 1]

tripleIntervals = [3, 1, 1, 2, 3, 5]

partialMeasures = [
    (1, 3),
    (1, 4),
    (2, 4),
    (5, 3),
    (3, 2),
    (7, 1)
]

intervalsList = [1, 3, 1, 2, 3]
noteCountsList = [4, 3, 5, 4, 3]
directionsList = [True, False, False, True, False]

miscMeasures = [
    (1, 3, True),
    (1, 4, False),
    (3, 4, True),
    (2, 1, True),
    (5, 2, False),
    (1, 3, False),
    (1, 7, True),
    (2, 5, False)
]

moreMeasures = [
    (1, 6, False),
    (2, 3, True),
    (2, 2, True),
    (3, 2, False),
]


#-------#
# Tests #
#-------#

freshTestSuite('makeRunsList')
tester = testFunctionMaybe(measureMunger, 'makeRunsList')
tester.case([3, 4, 2]).checkReturnValue(
    [(1, 3, True), (1, 4, True), (1, 2, True)]
)
tester.case([2, 3]).checkReturnValue(
    [(1, 2, True), (1, 3, True)]
)
tester.case(runLengths).checkReturnValue(
    [(1, notes, True) for notes in runLengths]  # shhhhh
)

freshTestSuite('makeTriplesList')
tester = testFunctionMaybe(measureMunger, 'makeTriplesList')
tester.case(True, [2, 3, 3]).checkReturnValue(
    [(2, 3, True), (3, 3, True), (3, 3, True)]
)
tester.case(False, tripleIntervals).checkReturnValue(
    [
        (3, 3, False),
        (1, 3, False),
        (1, 3, False),
        (2, 3, False),
        (3, 3, False),
        (5, 3, False),
    ]
)

freshTestSuite('makeListWithDirections')
tester = testFunctionMaybe(measureMunger, 'makeListWithDirections')
tester.case([(3, 4), (2, 1)], True).checkReturnValue(
    [(3, 4, True), (2, 1, True)]
)
tester.case(partialMeasures, False).checkReturnValue(
    [
        (1, 3, False),
        (1, 4, False),
        (2, 4, False),
        (5, 3, False),
        (3, 2, False),
        (7, 1, False)
    ]
)

freshTestSuite('makeMeasuresList')
tester = testFunctionMaybe(measureMunger, 'makeMeasuresList')
tester.case([5, 6], [2, 4], [True, False]).checkReturnValue(
    [(5, 2, True), (6, 4, False)]
)
tester.case(intervalsList, noteCountsList, directionsList).checkReturnValue(
    [
        (1, 4, True),
        (3, 3, False),
        (1, 5, False),
        (2, 4, True),
        (3, 3, False),
    ]
)

freshTestSuite('oddNotesMeasures')
tester = testFunctionMaybe(measureMunger, 'oddNotesMeasures')
tester.case([(3, 1, True), (5, 3, False)]).checkReturnValue(
    [(3, 1, True), (5, 3, False)]
)
tester.case([(4, 2, True), (6, 4, True)]).checkReturnValue([])
tester.case([(1, 7, False), (2, 5, False)]).checkReturnValue(
    [(1, 7, False), (2, 5, False)]
)

freshTestSuite('measuresInDirection')
tester = testFunctionMaybe(measureMunger, 'measuresInDirection')
tester.case([(4, 1, False), (6, 2, False)], True).checkReturnValue([])
tester.case([(4, 1, False), (6, 2, False)], False).checkReturnValue(
    [(4, 1, False), (6, 2, False)]
)
tester.case(miscMeasures, True).checkReturnValue(
    [
        (1, 3, True),
        (3, 4, True),
        (2, 1, True),
        (1, 7, True),
    ]
)
tester.case(miscMeasures, False).checkReturnValue(
    [
        (1, 4, False),
        (5, 2, False),
        (1, 3, False),
        (2, 5, False)
    ]
)

freshTestSuite('measuresThatReach')
tester = testFunctionMaybe(measureMunger, 'measuresThatReach')
tester.case(miscMeasures, wavesynth.A4).checkReturnValue(
    [
        (3, 4, True),
        (1, 7, True),
    ]
)
tester.case(miscMeasures, wavesynth.F6).checkReturnValue([])
tester.case(moreMeasures, wavesynth.D4).checkReturnValue(
    [
        (2, 3, True),
        (2, 2, True),
    ]
)
tester.case(moreMeasures, wavesynth.B4).checkReturnValue([(2, 3, True)])

freshTestSuite('getDirectionAndNotes')
tester = testFunctionMaybe(measureMunger, 'getDirectionAndNotes')
tester.case((4, 3, True)).checkReturnValue((True, 3))
tester.case((1, 10, False)).checkReturnValue((False, 10))

freshTestSuite('getPitchSpanIntervalAndNotes')
tester = testFunctionMaybe(measureMunger, 'getPitchSpanIntervalAndNotes')
tester.case((4, 3, True)).checkReturnValue((5.039684199579509, 4, 3))
tester.case((1, 10, False)).checkReturnValue((3.9999999999999956, 1, 10))

freshTestSuite('sortByInterval')
if hasattr(measureMunger, "sortByInterval"):
    tester = testBlock("""\
result = measures[:]  # make a copy
nope = measureMunger.sortByInterval(result, ascending)
""")
    case1 = tester.case(
        measureMunger=measureMunger,
        measures=miscMeasures,
        ascending=True
    )
    case1.checkVariableValue("nope", None)
    case1.checkVariableValue(
        "result",
        [
            (1, 3, False),
            (1, 3, True),
            (1, 4, False),
            (1, 7, True),
            (2, 1, True),
            (2, 5, False),
            (3, 4, True),
            (5, 2, False),
        ]
    )
    case2 = tester.case(
        measureMunger=measureMunger,
        measures=moreMeasures,
        ascending=True
    )
    case2.checkVariableValue("nope", None)
    case2.checkVariableValue(
        "result",
        [
            (1, 6, False),
            (2, 2, True),
            (2, 3, True),
            (3, 2, False),
        ]
    )
    case3 = tester.case(
        measureMunger=measureMunger,
        measures=moreMeasures[::-1],
        ascending=True
    )
    case3.checkVariableValue("nope", None)
    case3.checkVariableValue(
        "result",
        [
            (1, 6, False),
            (2, 2, True),
            (2, 3, True),
            (3, 2, False),
        ]
    )
    case4 = tester.case(
        measureMunger=measureMunger,
        measures=moreMeasures[::-1],
        ascending=False
    )
    case4.checkVariableValue("nope", None)
    case4.checkVariableValue(
        "result",
        [
            (3, 2, False),
            (2, 3, True),
            (2, 2, True),
            (1, 6, False),
        ]
    )

freshTestSuite('sortByDirection')
if hasattr(measureMunger, "sortByDirection"):
    tester = testBlock("""\
result = measures[:]  # make a copy
nope = measureMunger.sortByDirection(result)
""")
    case1 = tester.case(measureMunger=measureMunger, measures=miscMeasures)
    case1.checkVariableValue("nope", None)
    case1.checkVariableValue(
        "result",
        [
            (5, 2, False),
            (1, 3, False),
            (1, 4, False),
            (2, 5, False),
            (2, 1, True),
            (1, 3, True),
            (3, 4, True),
            (1, 7, True),
        ]
    )
    case2 = tester.case(measureMunger=measureMunger, measures=moreMeasures)
    case2.checkVariableValue("nope", None)
    case2.checkVariableValue(
        "result",
        [
            (3, 2, False),
            (1, 6, False),
            (2, 2, True),
            (2, 3, True),
        ]
    )

freshTestSuite('sortBySpan')
if hasattr(measureMunger, "sortBySpan"):
    tester = testBlock("""\
result = measures[:]  # make a copy
nope = measureMunger.sortBySpan(result)
""")
    case1 = tester.case(measureMunger=measureMunger, measures=miscMeasures)
    case1.checkVariableValue("nope", None)
    case1.checkVariableValue(
        "result",
        [
            (2, 1, True),
            (1, 3, True),
            (1, 3, False),
            (1, 4, False),
            (1, 7, True),
            (2, 5, False),
            (5, 2, False),
            (3, 4, True),
        ]
    )
    case2 = tester.case(measureMunger=measureMunger, measures=moreMeasures)
    case2.checkVariableValue("nope", None)
    case2.checkVariableValue(
        "result",
        [
            (2, 2, True),
            (2, 3, True),
            (1, 6, False),
            (3, 2, False),
        ]
    )


freshTestSuite('addMeasureSequence')
wavesynth.resetTracks()
if hasattr(measureMunger, "addMeasureSequence"):
    tester = testBlock("""\
wavesynth.resetTracks()
nope = measureMunger.addMeasureSequence(sequence, duration, base)
wavesynth.printTrack()
""", includeGlobals=True)
    case1 = tester.case(
        sequence=miscMeasures[:3],
        duration=0.2,
        base=wavesynth.C4
    )
    case1.checkVariableValue("nope", None)
    case1.checkPrintedLines(
        "a 0.2s keyboard note at C4 (60% vol)",
        "and a 0.2s keyboard note at D4 (60% vol)",
        "and a 0.2s keyboard note at E4 (60% vol)",
        "and a 0.2s keyboard note at G4 (60% vol)",
        "and a 0.2s keyboard note at E4 (60% vol)",
        "and a 0.2s keyboard note at D4 (60% vol)",
        "and a 0.2s keyboard note at C4 (60% vol)",
        "and a 0.2s keyboard note at A3 (60% vol)",
        "and a 0.2s keyboard note at E4 (60% vol)",
        "and a 0.2s keyboard note at C5 (60% vol)",
        "and a 0.2s keyboard note at G5 (60% vol)"
    )
    case2 = tester.case(
        sequence=moreMeasures[::-1],
        duration=0.1,
        base=wavesynth.A4
    )
    case2.checkVariableValue("nope", None)
    case2.checkPrintedLines(
        "a 0.1s keyboard note at A4 (60% vol)",
        "and a 0.1s keyboard note at D4 (60% vol)",
        "and a 0.1s keyboard note at G3 (60% vol)",
        "and a 0.1s keyboard note at C4 (60% vol)",
        "and a 0.1s keyboard note at E4 (60% vol)",
        "and a 0.1s keyboard note at A4 (60% vol)",
        "and a 0.1s keyboard note at D5 (60% vol)",
        "and a 0.1s keyboard note at G5 (60% vol)",
        "and a 0.1s keyboard note at E5 (60% vol)",
        "and a 0.1s keyboard note at D5 (60% vol)",
        "and a 0.1s keyboard note at C5 (60% vol)",
        "and a 0.1s keyboard note at A4 (60% vol)",
        "and a 0.1s keyboard note at G4 (60% vol)"
    )

showOverview()
