# Your name: Peter Mawhorter
# Your username: pmwh
# CS111 PS04 Task 2 Supporting Code
# maze.py
# Created: 2019/09/23

# time module for waiting to control animation speed
import time

# for the randomMove function
import random

# turtle for drawing the maze state
from turtle import (
    color, pensize, penup, goto, pendown, setheading, showturtle,
    hideturtle, stamp, begin_fill, end_fill, fd, bk, rt, lt, circle,
    write, reset, tracer, update
)

#--------------------------------------#
# Constants and other global variables #
#--------------------------------------#

# Maximum number of steps allowed
# 150 is too few for some of the larger mazes, use 200 or 300 instead.
DEFAULT_MAX_STEPS = 150

# Speed in Hertz (updates per second):
# Use 0 to update as quickly as possible
DEFAULT_SPEED = 1.5

# These values are used to determine directions, and should be returned by
# policy functions.
FORWARD = "FORWARD"
RIGHT = "RIGHT"
BACKWARD = "BACKWARD"
LEFT = "LEFT"

# All directions in a list
DIRECTIONS = [FORWARD, RIGHT, BACKWARD, LEFT]

# Several different mazes of varying difficulty:
MAZE0 = """\
######
#>  G#
######\
"""

MAZE1 = """\
######
#    #
# ## #
#G#> #
######\
"""

MAZE2 = """\
#######
#     #
# ### #
#  <#G#
#######\
"""

MAZE3 = """\
#######
#v#   #
# # # #
#   #G#
#######\
"""

MAZE4 = """\
#########
# # #G# #
# # # # #
#   <   #
#########\
"""

MAZE5 = """\
#########
# # #G# #
# # # # #
#   >   #
#########\
"""

MAZE6 = """\
#########
# #     #
#   # # #
# ### ###
# ^#   G#
#########\
"""

MAZE7 = """\
##################
#v#     #   ##   #
# # ###   #    # #
#     ##### # ## #
# ### #     #  ###
#   ### ######   #
# # ##   #   ### #
# # G# # # #     #
##################\
"""

MAZE8 = """\
##########
#        #
#    v#  #
# G#     #
#        #
##########\
"""

MAZE9 = """\
#######
#     #
#     #
#  G  #
#     #
#^    #
#######\
"""

MAZE10 = """\
#########
#>      #
#       #
#       #
##     ##
###   ###
### G ###
###   ###
#########\
"""

MAZE11 = """\
###########
###     ###
###     ###
###  G  ###
###     ###
###     ###
### ### ###
###     ###
##       ##
#         #
#         #
#>        #
###########\
"""


MAZE12 = """\
###############
#             #
#   #######   #
#             #
# # # ### # # #
# #         # #
# # #     # # #
# # #  G  # # #
# # #     # # #
# #         # #
# # # ### # # #
#             #
#   #######   #
#>            #
###############\
"""


#------------------#
# Helper functions #
#------------------#

def rightOf(direction):
    """
    Returns the direction that's to the right of the given direction.

    Raises a `ValueError` if given an invalid direction.
    """
    if direction not in DIRECTIONS:
        raise ValueError("Invalid direction '{}'.".format(direction))
    idx = DIRECTIONS.index(direction)
    return DIRECTIONS[(idx + 1) % len(DIRECTIONS)]


def leftOf(direction):
    """
    Returns the direction that's to the left of the given direction.

    Raises a `ValueError` if given an invalid direction.
    """
    if direction not in DIRECTIONS:
        raise ValueError("Invalid direction '{}'.".format(direction))
    idx = DIRECTIONS.index(direction)
    return DIRECTIONS[(idx + 3) % len(DIRECTIONS)]


def reverseOf(direction):
    """
    Returns the direction that's the reverse of the given direction.

    Raises a `ValueError` if given an invalid direction.
    """
    if direction not in DIRECTIONS:
        raise ValueError("Invalid direction '{}'.".format(direction))
    idx = DIRECTIONS.index(direction)
    return DIRECTIONS[(idx + 2) % len(DIRECTIONS)]


def randomDirection():
    """
    Returns a random direction.
    """
    return random.choice(DIRECTIONS)


def noWallInDirection(direction, ahead, right, behind, left):
    """
    Returns True if there's not a wall in the given direction.
    """
    if direction == FORWARD and ahead != '#':
        return True
    elif direction == RIGHT and right != '#':
        return True
    elif direction == LEFT and left != '#':
        return True
    elif direction == BACKWARD and behind != '#':
        return True
    else:
        return False


def randomNonWallDirection(ahead, right, behind, left):
    """
    Returns a random direction that doesn't contain a wall. Not a
    perfectly even distribution over each possibility (directions closer
    to blocked directions are more likely).

    Returns a random direction if everything is blocked.
    """
    check = random.choice(DIRECTIONS)
    turn = random.choice([leftOf, rightOf])
    if noWallInDirection(check, ahead, right, behind, left):
        return check

    check = turn(check)
    if noWallInDirection(check, ahead, right, behind, left):
        return check

    check = turn(check)
    if noWallInDirection(check, ahead, right, behind, left):
        return check

    check = turn(check)
    if noWallInDirection(check, ahead, right, behind, left):
        return check

    # Everything is blocked...
    return randomDirection()

    # Concise implementation w/ lists and loops:
#    directions = [FORWARD, RIGHT, BACKWARD, LEFT]
#    blocked = [x == '#' for x in [ahead, right, behind, left]]
#    random.shuffle(directions)
#    for d in directions:
#        if not blocked[d]:
#            return d
#    return randomDirection()


def coordsInDirection(x, y, d):
    """
    Accepts x/y coordinates and a direction and returns the coordinates of
    the tile that's in that direction from the given coordinates.
    """
    if d == FORWARD:
        return [x, y - 1]
    elif d == RIGHT:
        return [x + 1, y]
    elif d == BACKWARD:
        return [x, y + 1]
    elif d == LEFT:
        return [x - 1, y]
    else:
        raise ValueError("Invalid direction: " + repr(d))


def isCompleted(maze):
    """
    A maze is completed if it has the 'S' for success character anywhere
    inside.
    """
    return 'S' in maze


def robotFacing(direction):
    """
    Returns the character which represents a robot facing the given
    direction.
    """
    if direction == FORWARD:
        return '^'
    elif direction == RIGHT:
        return '>'
    elif direction == BACKWARD:
        return 'v'
    elif direction == LEFT:
        return '<'


#--------------------------#
# Complex helper functions #
#--------------------------#

def getSurroundings(maze):
    """
    Looks at the given maze and finds the robot in it using
    getRobotPositionAndFacing. Then figures out what tiles are situated on
    all sides of the robot, and returns the tiles to the front, right,
    back, and left of the robot in that order.
    """
    x, y, f = getRobotPositionAndFacing(maze)

    # Get tiles from each direction relative to robot
    frontX, frontY = coordsInDirection(x, y, f)
    inFront = getMazeTileAt(maze, frontX, frontY)

    leftX, leftY = coordsInDirection(x, y, leftOf(f))
    atLeft = getMazeTileAt(maze, leftX, leftY)

    rightX, rightY = coordsInDirection(x, y, rightOf(f))
    atRight = getMazeTileAt(maze, rightX, rightY)

    behindX, behindY = coordsInDirection(x, y, reverseOf(f))
    inBack = getMazeTileAt(maze, behindX, behindY)

    return [ inFront, atRight, inBack, atLeft ]


def getMazeTileAt(maze, x, y):
    """
    Gets the tile from a maze at specific x/y coordinates.
    """
    rows = maze.split('\n')
    if 0 <= y < len(rows):
        row = rows[y]
        if 0 <= x <= len(row):
            return row[x]
        else:
            return '#'
    else:
        return '#' # tiles outside the maze are walls


def getRobotPositionAndFacing(maze):
    """
    Looks at a maze and finds a robot in it, returning the coordinates and
    facing of the robot. FORWARD is used for North, with the other
    directions being relative to that.
    """
    rows = maze.split('\n')
    for y, row in enumerate(rows):
        for x, char in enumerate(row):
            if char == '^':
                return [x, y, FORWARD]
            elif char == '>':
                return [x, y, RIGHT]
            elif char == 'v':
                return [x, y, BACKWARD]
            elif char == '<':
                return [x, y, LEFT]
            # else continue searching for the robot in the maze
    # Unable to find robot?!?!
    raise ValueError("No robot in the maze!")


#----------------#
# Core functions #
#----------------#

def showSurroundings(maze, step=None):
    """
    Prints a line of text that summarizes the robot's current
    surroundings, including the step count if one is given.
    """
    if step is not None:
        print("Step {}: ".format(step), end="")

    try:
        around = getSurroundings(maze)
        facing = "unknown"
        offset = 0
        if '^' in maze:
            facing = "north"
            offset = 0
        elif '>' in maze:
            facing = "east"
            offset = -1
        elif 'v' in maze:
            facing = "south"
            offset = -2
        elif '<' in maze:
            facing = "west"
            offset = -3

        facing = "Facing {}.".format(facing)

        walls = []
        visited = []
        goal = None
        for i, ori in enumerate([ "north", "east", "south", "west" ]):
            what = around[(offset + i) % 4]
            if what == "#":
                walls.append(ori)
            elif what == ".":
                visited.append(ori)
            elif what == "G":
                goal = ori

        if len(walls) == 0:
            walls = "No walls."
        elif len(walls) == 1:
            walls = "One wall {}.".format(*walls)
        elif len(walls) == 2:
            walls = "Two walls {} and {}.".format(*walls)
        elif len(walls) == 3:
            walls = "Three walls {}, {}, and {}.".format(*walls)
        else:
            walls = "Walls on all sides."

        if len(visited) == 0:
            visited = None
        elif len(visited) == 1:
            visited = "Visited {}.".format(*visited)
        elif len(visited) == 2:
            visited = "Visited {} and {}.".format(*visited)
        elif len(visited) == 3:
            visited = "Visited {}, {}, and {}.".format(*visited)
        else:
            visited = "Already visited everywhere nearby."

        if goal is not None:
            goal = "The goal is to the {}!".format(goal)

        print(
            ' '.join(
                [d for d in [facing, walls, visited, goal] if d is not None]
            )
        )
    except ValueError:
        if 'S' in maze:
            print("The robot has reached the goal.")
        else:
            print("There is no robot in the maze.")


def showMaze(maze, step=None):
    """
    Displays a maze by printing it, and if a step value is supplied also
    prints that.
    """
    print()
    if step is not None:
        print("Step " + str(step))
    print(maze)
    print()


def traceSurroundings(
    maze,
    policy,
    fuel=DEFAULT_MAX_STEPS,
    speed=DEFAULT_SPEED
):
    """
    Prints a description of the robot's surroundings as the given policy
    moves through the maze. The speed parameter controls how fast the
    frames are printed, in frames drawn per second. Use 0 for speed to
    draw as quickly as possible. The fuel parameter controls how many
    steps the robot may take. A message will be printed if the robot
    takes too many steps.
    """
    print(
        "Tracing surroundings with {} fuel in the following maze:".format(
            fuel
        )
    )
    print(maze)
    print()
    for step in range(fuel):
        fuelRemaining = fuel - step
        showSurroundings(maze, step)

        # Check the action so we can display it
        ft, rt, bk, lt = getSurroundings(maze)
        action = policy(ft, rt, bk, lt, fuelRemaining)
        if action in DIRECTIONS:
            print("  Decision: {}.".format(action))
        else:
            print(
                "  Policy returned an invalid value: {}".format(repr(action))
            )
        print()

        # Delay according to the speed
        if speed > 0:
            time.sleep(1 / float(speed))
        elif speed == "pause":
            input("Step {}. [press enter]".format(step))
        # else no delay

        maze = nextMazeState(maze, policy, fuelRemaining)
        if isCompleted(maze):
            showSurroundings(maze, step + 1)
            print("Found the goal! Hooray!")
            break

    if not isCompleted(maze):
        print("Ran out of fuel.")

    print()
    print("The final maze state is:")
    print(maze)


def animateText(maze, policy, fuel=DEFAULT_MAX_STEPS, speed=DEFAULT_SPEED):
    """
    Prints repeated frames of the maze state as the given policy moves
    through the maze. The speed parameter controls how fast the frames are
    printed, in frames drawn per second. Use 0 for speed to draw as
    quickly as possible. The fuel parameter controls how many steps the
    robot may take. A message will be printed if the robot takes too many
    steps.
    """
    print("Animating maze movement with {} fuel:".format(fuel))
    print()
    for step in range(fuel):
        fuelRemaining = fuel - step
        showMaze(maze, step)
        if speed > 0:
            time.sleep(1 / float(speed))
        elif speed == "pause":
            input("Step {}. [press enter]".format(step))
        # else no delay
        maze = nextMazeState(maze, policy, fuelRemaining)
        if isCompleted(maze):
            showMaze(maze, step + 1)
            print("Found the goal! Hooray!")
            break

    if not isCompleted(maze):
        print("Ran out of fuel.")


def testPolicy(maze, policy, fuel=DEFAULT_MAX_STEPS):
    """
    Tests whether a policy can make it to the goal within a specified
    number of steps. Works like animateText, but just returns True (if the
    robot makes it to the goal) or False (if not) without printing
    anything.
    """
    # +1 here so that fuel = 1 tests after 1 step
    for step in range(fuel + 1):
        fuelRemaining = fuel - step
        if isCompleted(maze):
            return True
        maze = nextMazeState(maze, policy, fuelRemaining)

    return False # ran out of steps


def testPolicyThoroughly(maze, policy, fuel=DEFAULT_MAX_STEPS, trials=500):
    """
    Tests a policy many times on the given maze, and returns the
    percentage of attempts that succeeded as a number between 0 and 1
    (inclusive). Uses 500 trials by default.
    """
    successes = 0
    for i in range(trials):
        if testPolicy(maze, policy, fuel=fuel):
            successes += 1

    return successes / trials


def nextMazeState(maze, policy, fuelRemaining):
    """
    This function takes a maze with a robot at a particular position and
    returns a new maze where that robot has moved once. The given policy
    function determines how the robot moves: it will be called with the
    front, right, back, and left characters adjacent to the robot, as
    well as the amount of fuel remaining, and should return one of the
    directional constants maze.FORWARD, maze.RIGHT, maze.BACKWARD, or
    maze.LEFT. The robot will turn to face the given direction and move
    one square in that direction, or stay still if there is a wall in
    that direction.
    """
    # Get robot position, facing, and surrounding tiles:
    x, y, f = getRobotPositionAndFacing(maze)
    ft, rt, bk, lt = getSurroundings(maze)

    action = policy(ft, rt, bk, lt, fuelRemaining)

    # figure out new facing based o policy action:
    if action == FORWARD:
        newF = f
    elif action == RIGHT:
        newF = rightOf(f)
    elif action == BACKWARD:
        newF = reverseOf(f)
    elif action == LEFT:
        newF = leftOf(f)
    else:
        raise ValueError(
            "Invalid action: " + repr(action) + """
(the action returned by an policy must be one of maze.FORWARD, maze.RIGHT,
maze.BACKWARD, or maze.LEFT)"""
        )

    # compute new position and figure out what's there
    newX, newY = coordsInDirection(x, y, newF)
    atNew = getMazeTileAt(maze, newX, newY)

    if atNew in '^>v<#': # the robot or blocked: turn without moving
        return mazeWithUpdate(maze, x, y, robotFacing(newF))
    elif atNew == 'G': # the goal: turn into an S for success
        return mazeWithUpdate(mazeWithUpdate(maze, x, y, '.'), newX, newY, 'S')
    else: # an empty spot: move there
        rob = robotFacing(newF)
        return mazeWithUpdate(mazeWithUpdate(maze, x, y, '.'), newX, newY, rob)


def mazeWithUpdate(maze, x, y, newTile):
    """
    Accepts a maze and returns a new maze where the tile at the given x/y
    coordinates has been replaced with the given new tile. Makes no change
    if the location is outside the maze.
    """
    rows = maze.split('\n')
    result = ''
    for rowY, row in enumerate(rows):
        for tileX, tile in enumerate(row):
            if rowY == y and tileX == x:
                result = result + newTile
            else:
                result = result + tile
        # After every full cycle of the inner loop (except the last), we add
        # a newline to our result
        if rowY < len(rows) - 1:
            result = result + '\n'

    return result


#-------------------#
# Drawing functions #
#-------------------#

# Size of a maze cell in the turtle window:
MAZE_GRID_SIZE = 30


def drawEachTile(maze, artist):
    """
    Moves the turtle through each tile position in the maze and calls the
    given artist function with the maze character at that position.
    """
    pensize(1)

    rows = maze.split('\n')
    height = len(rows)
    width = max(len(row) for row in rows)

    adjx = -MAZE_GRID_SIZE * (width / 2)
    adjy = -MAZE_GRID_SIZE * (height / 2)

    for rowidx, row in enumerate(rows):
        # Invert y axis to match printed order:
        y = height - 1 - rowidx
        for x, char in enumerate(row):
            # Go to upper-left of square:
            penup()
            goto(
                MAZE_GRID_SIZE * (x - 0.5) + adjx,
                MAZE_GRID_SIZE * (y + 0.5) + adjy
            )
            pendown()
            setheading(0)

            # Call the artist
            artist(char)


def wallsArtist(char):
    """
    Artist to draw walls & goal for the maze.
    """
    if char == '#':
        # Draw a filled black square:
        color("black")
        begin_fill()
        fd(MAZE_GRID_SIZE)
        rt(90)
        fd(MAZE_GRID_SIZE)
        rt(90)
        fd(MAZE_GRID_SIZE)
        rt(90)
        fd(MAZE_GRID_SIZE)
        rt(90)
        end_fill()

    elif char == 'G':
        # Go to upper-mid of square:
        penup()
        fd(MAZE_GRID_SIZE / 2)
        rt(90)
        fd(0.15 * MAZE_GRID_SIZE)
        lt(90)
        pendown()
        # Draw a filled green circle:
        color("SpringGreen3")
        begin_fill()
        circle(-0.35 * MAZE_GRID_SIZE)
        end_fill()


def solvedArtist(char):
    """
    Artist to draw the solution of a solved maze.
    """
    if char == 'S':
        # Go to upper-left of square:
        penup()
        fd(0.3 * MAZE_GRID_SIZE)
        rt(90)
        fd(0.15 * MAZE_GRID_SIZE)
        lt(90)
        pendown()
        # Draw a filled blue rectangle w/ rounded corners:
        color("SkyBlue4")
        begin_fill()
        fd(0.4 * MAZE_GRID_SIZE)
        circle(-0.15 * MAZE_GRID_SIZE, 90)
        fd(0.4 * MAZE_GRID_SIZE)
        circle(-0.15 * MAZE_GRID_SIZE, 90)
        fd(0.4 * MAZE_GRID_SIZE)
        circle(-0.15 * MAZE_GRID_SIZE, 90)
        fd(0.4 * MAZE_GRID_SIZE)
        circle(-0.15 * MAZE_GRID_SIZE, 90)
        end_fill()


def drawMazeWalls(maze):
    """
    Uses turtle commands to draw the structure of a maze, with black
    rectangles for the walls and a green circle for the goal. Does not draw
    the robot or any visited regions.
    """
    drawEachTile(maze, wallsArtist)


def drawSolution(maze):
    """
    Draws a blue square at the solution of a solved maze.
    """
    drawEachTile(maze, solvedArtist)


def putTurtleAt(xy, maze):
    """
    Moves the turtle to the upper-left corner of the given (x, y)
    position and orients it facing due East. The y-axis used for input
    coordinates is the maze y-axis, with +y to the South (string index
    coordinates).
    """
    x, rowidx = xy

    # Compute adjx and adjy
    rows = maze.split('\n')
    height = len(rows)
    width = max(len(row) for row in rows)

    y = height - 1 - rowidx

    adjx = -MAZE_GRID_SIZE * (width / 2)
    adjy = -MAZE_GRID_SIZE * (height / 2)

    # Go to upper-mid of square:
    penup()
    goto(
        MAZE_GRID_SIZE * (x - 0.5) + adjx,
        MAZE_GRID_SIZE * (y + 0.5) + adjy
    )
    pendown()
    setheading(0)


def drawVisited(rpos, maze):
    """
    Draws a visited dot at the given robot location (x, y, f).
    """
    x, y, f = rpos

    # Go to tile
    putTurtleAt((x, y), maze)

    # draw a large white square to erase robot stamp:
    color("white")
    penup()
    fd(0.1 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.1 * MAZE_GRID_SIZE)
    lt(90)
    begin_fill()
    fd(0.8 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.8 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.8 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.8 * MAZE_GRID_SIZE)
    rt(90)
    end_fill()
    bk(0.1 * MAZE_GRID_SIZE)
    lt(90)
    fd(0.1 * MAZE_GRID_SIZE)
    rt(90)
    pendown()

    # Go to upper-mid of tile:
    penup()
    fd(0.5 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.4 * MAZE_GRID_SIZE)
    lt(90)
    pendown()

    # Draw a small blue circle:
    color("SkyBlue3")
    pensize(1)
    begin_fill()
    circle(-MAZE_GRID_SIZE * 0.1)
    end_fill()


def drawRobotPosition(rpos, maze):
    """
    Draws a turtle stamp on the given robot position (x, y, f). Will be
    drawn over by the visited dot as part of the next drawing step.
    """
    x, y, f = rpos

    # Go to tile
    putTurtleAt((x, y), maze)

    # Go to center of tile:
    penup()
    fd(0.5 * MAZE_GRID_SIZE)
    rt(90)
    fd(0.5 * MAZE_GRID_SIZE)
    pendown()

    # Set heading:
    setheading(90 - 90 * DIRECTIONS.index(f))

    # Draw stamp
    color("black")
    showturtle()
    stamp()


SBOX_POS = (-80, 310)
SBOX_SIZE = (160, 40)


def displayStepCounter(steps):
    """
    Displays the step counter using text.
    """
    textInBox("Step #{}".format(steps), SBOX_POS, SBOX_SIZE)


MBOX_POS = (-180, -270)
MBOX_SIZE = (360, 40)


def displayMessage(msg):
    """
    Redraws the message box and displays a message.
    """
    textInBox(msg, MBOX_POS, MBOX_SIZE)


def textInBox(msg, boxPos, boxSize):
    """
    Draws text in a black box. Fills white over anything previously in
    the box.
    """
    # Set pen parameters:
    pensize(1)
    color("black", "white")
    setheading(0)

    # Go to step box position:
    penup()
    goto(*boxPos)
    pendown()

    # Draw filled square
    begin_fill()
    fd(boxSize[0])
    rt(90)
    fd(boxSize[1])
    rt(90)
    fd(boxSize[0])
    rt(90)
    fd(boxSize[1])
    rt(90)
    end_fill()

    # Go to bottom middle of box:
    penup()
    fd(boxSize[0] / 2)
    rt(90)
    fd(boxSize[1] * 0.8)

    # Draw center-aligned text:
    write(msg, align='center', font=('Arial', 22, 'normal'))


def animateTurtle(maze, policy, fuel=DEFAULT_MAX_STEPS, speed=DEFAULT_SPEED):
    """
    Animates the robot moving through the maze, drawing small light blue
    circles for visited areas. The turtle is used to "draw" the robot.
    Begins by resetting, disabling tracing, and drawing the maze walls
    once.

    The speed parameter controls how fast the frames are drawn, in frames
    drawn per second. Use 0 for speed to draw as quickly as possible. The
    fuel parameter controls How many steps the robot is allowed to take.
    Each frame calls update.

    A message will be displayed if the robot takes too many steps.
    """
    reset()
    tracer(0, 0)
    drawMazeWalls(maze) # draw walls once
    rpos = None
    for step in range(fuel):
        fuelRemaining = fuel - step
        if rpos:
            drawVisited(rpos, maze) # draw visited dot on previous position
        rpos = getRobotPositionAndFacing(maze)
        drawRobotPosition(rpos, maze) # draw new robot position
        displayStepCounter(step)
        displayMessage("Fuel remaining: {}".format(fuelRemaining))
        hideturtle()
        update()
        if speed > 0:
            time.sleep(1 / float(speed))
        maze = nextMazeState(maze, policy, fuelRemaining)
        if isCompleted(maze):
            drawVisited(rpos, maze) # draw visited dot over last position
            drawSolution(maze)
            displayStepCounter(step + 1)
            displayMessage("Found the goal! Hooray!")
            hideturtle()
            update()
            if speed > 0: # Linger briefly on solved view:
                time.sleep(max(1 / float(speed), 0.5))
            break

    if not isCompleted(maze):
        displayMessage("Ran out of fuel. :(")
        hideturtle()
        if speed > 0: # Linger briefly on failure message:
            time.sleep(max(1 / float(speed), 0.5))
        update()


#--------------#
# Testing code #
#--------------#

# Super-secret tech:
def randomUnvisitedDirection(ahead, right, behind, left):
    """
    Returns a random direction that doesn't contain a wall and which
    hasn't been visited.
    """
    # Note: it's possible but tedious to implement this without lists or
    # loops (see randomNonWallDirection above).
    directions = DIRECTIONS[:] # the [:] makes a copy
    blocked = [x in '#.' for x in [ahead, right, behind, left]]
    random.shuffle(directions)
    for d in directions:
        if not blocked[DIRECTIONS.index(d)]:
            return d
    return randomNonWallDirection(ahead, right, behind, left)


def turnAroundPolicy(ahead, right, behind, left, fuel):
    """
    The turnAroundPolicy always moves forward unless there's a wall ahead,
    in which case it turns around. It is very bad at solving mazes.
    """
    if ahead != '#': # there's no wall in front of us: go forward
        return FORWARD # keep going forward
    else: # otherwise turn around
        return BACKWARD # tell the robot to go backwards


def attentiveVersion(basePolicy):
    """
    Create an attentive version of any other policy which will
    immediately go to the goal if it is adjacent.
    """
    def modified(ahead, right, behind, left, fuel):
        """
        An attentive policy that falls back to some base policy behavior
        if it can't see the goal.
        """
        if ahead == 'G':
            return FORWARD
        elif left == 'G':
            return LEFT
        elif right == 'G':
            return RIGHT
        elif behind == 'G':
            return BACKWARD
        else:
            return basePolicy(ahead, right, behind, left, fuel)

    return modified


def smartRandomPolicy(ahead, right, behind, left, fuel):
    """
    A policy that randomly explores unvisited space, and falls back to a
    bounce policy if it's surrounded by visited spaces, with an extra bit
    of randomness thrown in every 7 steps.
    """
    if ahead == ' ' or right == ' ' or behind == ' ' or left == ' ':
        return randomUnvisitedDirection(ahead, right, behind, left)
    elif fuel % 15 == 0: # extra random every 7 steps
        return randomNonWallDirection(ahead, right, behind, left)
    elif left != '#':
        return LEFT
    elif ahead != '#':
        return FORWARD
    elif right != '#':
        return RIGHT
    else:
        return BACKWARD


def test():
    """
    Main testing code:
    """
    trialPolicy = attentiveVersion(smartRandomPolicy)

    for (maze, fuel) in [
        (MAZE0, 50),
        (MAZE1, 50),
        (MAZE2, 50),
        (MAZE3, 50),
        (MAZE4, 50),
        (MAZE5, 50),
        (MAZE6, 150),
        (MAZE7, 200),
        (MAZE8, 200),
        (MAZE9, 200),
        (MAZE10, 200),
        (MAZE11, 300),
        (MAZE12, 300),
    ]:
        traceSurroundings(maze, trialPolicy, fuel=fuel, speed=0)
        animateText(maze, trialPolicy, fuel=fuel, speed=0)
        animateTurtle(maze, trialPolicy, fuel=fuel, speed=25)
        print('-' * 80)
        print(maze)
        pct = testPolicyThoroughly(maze, trialPolicy, fuel)
        print(
            "With {} fuel, the policy solves this maze {:0.2f}% of the time."
            .format(fuel, 100 * pct)
        )


if __name__ == "__main__":
    test()
