"""
Authors: Peter Mawhorter
Consulted:
Date: 2022-1-2
Purpose: Drums melody example
  Demonstrates melody functions for mixedUpMelodies task, with drums.
"""

from wavesynth import *


def staggeredNoteBeat(duration):
    """
    Adds a note and a staggered beat that starts half the note's duration
    late. Leaves the track at the end of the note in terms of time.
    """
    addNote(duration)
    rewind(duration/2)
    quieter(3)
    addBeat(duration/2)
    louder(3)


def noteBeat(duration):
    """
    Adds a note and a beat at the same time that's half the duration of
    the note, and three steps quieter.
    """
    addNote(duration)
    rewind(duration)
    quieter(3)
    addBeat(duration/2)
    louder(3)
    fastforward(duration/2)


def noteDoubleBeat(duration):
    """
    Adds a note with two half/duration beats in the background.
    """
    addNote(duration)
    rewind(duration)
    quieter(3)
    addBeat(duration/2)
    addBeat(duration/2)
    louder(3)


def drumsIntro():
    """
    A drums intro melody. Leaves the pitch and volume at the same place
    it started.
    """
    noteDuration = 0.25

    # Three staggered notes + beats
    staggeredNoteBeat(noteDuration)
    halfStepUp(4)
    staggeredNoteBeat(noteDuration)
    halfStepDown(4)

    # A longer note w/ two short beats at the end
    addNote(noteDuration*2)
    rewind(noteDuration)
    quieter(3)
    addBeat(noteDuration/2)
    louder(3)
    fastforward(noteDuration/2)


def drumsRefrain():
    """
    A drums refrain. The last note isn't the same as the first one, but
    this function returns the current pitch value back to where it
    started in the end, and it also leaves the volume unchanged.
    """
    noteDuration = 0.25

    # Note/beat combos moving down the parts of a major chord and then
    # back to the original pitch
    noteBeat(noteDuration)
    halfStepDown(3)
    noteBeat(noteDuration)
    halfStepDown(4)
    noteBeat(noteDuration)
    halfStepUp(7)
    noteBeat(noteDuration)


def drumsEnding():
    """
    A drums ending. Ends on the same note it started on, with the volume
    unchanged. Total duration is 2 seconds.
    """
    noteDuration = 0.25

    # Two notes w/ double-beats followed by rests, 4 half-steps apart
    noteDoubleBeat(noteDuration)
    addRest(noteDuration)

    halfStepUp(4)
    noteDoubleBeat(noteDuration)
    addRest(noteDuration)

    # A 2x-duration note-double-beat at the original pitch
    halfStepDown(4)
    noteDoubleBeat(noteDuration*2)
