"""
Author: 
Date: 
Description:

"""
import csv
import unicodedata
import re
import random

def validate(q):
	"""Validates user responses"""
	repeat = True
	while repeat:
		ans = input(q)
		if positive_response(ans):
			return True
		elif negative_response(ans):
			return False
		else:
			print("Sorry, I didn't understand that.")

def get_poem_data():
	"""Preprocesses poem dataset"""
	pf = "kaggle_poem_dataset.csv"
	with open(pf,'r', encoding='utf-8') as csvf:
		reader = csv.DictReader(csvf)
		poems = [r for r in reader]
	for i,poem in enumerate(poems):
		# Normalize weird characters:
		poem['Content'] = unicodedata.normalize('NFKC', poem['Content'])
	return poems

def greet():
	"""Greets user and returns user's name"""
	print("Hello! I am Odette, a poetry-recommending chatbot.")
	name = input("What is your name? ")
	if " " in name:
		if "name is" in name:  	#Handles "my name is Carolyn"
			return name.split("name is ")[1] 
		else: 					#Handles "Carolyn Anderson"
			return name.split()[0]
	return name

def find_items(text):
	"""Finds list items in a user response string"""
	bits = [w.strip() for w in re.split(r',|and|or|;|\.',text)] #In-class mistake: text.strip() instead of w!
	return [b for b in bits if b]

def remove_verbs(s,mood):
	"""Removes preference verbs"""
	if mood == "pos":
		v_list = ["like","love","adore"]
	else:
		v_list = ["dislike", "don't like", "hate", "despise"]
	for w in v_list:
		if w in s:
			s = s.split(w)[1]
	return s

def theme(name):
	"""Queries the user's topic likes and dislikes and returns"""
	print("To help me pick a poem for you, tell me a little more about what you're interested in.")
	pos_text = input("What are some things that you like? ")
	pos_text = remove_verbs(pos_text,"pos")
	likes = find_items(pos_text)
	neg_text = input("What are some things that you dislike? ")
	neg_text = remove_verbs(neg_text,"neg")
	dislikes = find_items(neg_text)
	print(name,"likes:",', '.join(likes))
	print(name,"dislikes:",', '.join(dislikes))
	return likes, dislikes

def negative_response(s):
	"""Validates a negative response"""
	return True if re.match(r'[Nn]o',s) else False

def positive_response(s):
	"""Validates a positive response"""
	return True if re.match(r'[Yy]es',s) else False

def get_poets(s,mood):
	"""Gets liked or disliked poets"""
	poet_s = re.sub(r'[Yy]es[,!.:;] ','',s) #Clean start of response like, "Yes, Dickinson"
	poet_s = remove_verbs(poet_s,mood)
	return find_items(poet_s)

def poets():
	"""Queries user's poet preferences and returns"""
	liked_poets = []
	print("Do you have any preferences for or against certain poets?")
	liked = input("Are there any poets you particularly like? ")
	if not negative_response(liked):
		liked_poets = get_poets(liked,"pos")

	return liked_poets

def find_thematic_poems(theme,poems):
	"""Retrieves a set of poems with a particular theme, sorted by theme mention frequency"""
	thematic_poems = []
	for poem in poems:
		if ' '+theme+' ' in poem['Content']:  #forget to add spaces first
			thematic_poems.append(poem)
			find_idx = poem['Content'].find(theme)
	# Rank by number of mentions of theme:
	return sorted(thematic_poems,key=lambda x: -len([w for w in x['Content'].split() if w == theme]))

def remove_dislikes(dislikes,poems):
	"""Filters out poems that mention disliked themes"""
	filtered_poems = []
	for poem in poems:
		exclude = False
		for d in dislikes:
			if d in poem['Content']:
				exclude = True
		if not exclude:
			filtered_poems.append(poem)

	return filtered_poems if len(filtered_poems) > 0 else poems

def prioritize_authors(likes,poems):
	"""Sorts poems by whether they are by a preferred author"""
	filtered_poems = []
	for poem in poems:
		include = False
		for d in likes:
			if d.strip().lower() in poem['Author'].lower():
				include = True
		if include:
			filtered_poems.append(poem)
	if len(filtered_poems) == 0:
		return poems
	return filtered_poems

def poem_search(poems,dislikes,liked_poets):
	"""Applies user preferences to poem dataset"""
	filtered_poems = remove_dislikes(dislikes,poems)
	#print(len(filtered_poems))
	filtered_poems = prioritize_authors(liked_poets,filtered_poems)
	#print(len(filtered_poems))
	return filtered_poems

def print_poem(poem):
	"""Pretty-prints poem"""
	print(f"This is a poem by {poem['Author']} called {poem['Title']}.")
	print('~'*50)
	print(poem['Title'].upper())
	print('-'*40)
	for l in poem['Content'].split('\n'):
		print(l)
	print('~'*50)

def main():
	poems = get_poem_data()
	name = greet()
	likes, dislikes = theme(name)
	liked_poets = poets()

	random.shuffle(likes)
	new_poems = poems
	while len(new_poems) > 1 and len(likes) > 0: # Keep filtering while there are more theme preferences
		topic = likes.pop()
		theme_poems = find_thematic_poems(topic,new_poems)
		if len(theme_poems) > 1: # Give up on theme if there are no poems that fit it
			new_poems = theme_poems
			print(f"I'm searching for poems about {topic}.")

	goOn = True
	selections = poem_search(theme_poems,dislikes,liked_poets)
	s = 's' if len(selections) > 1 else ''
	print(f"I found {len(selections)} poem{s} that you might like.")
	if s:
		print("Here is one of them.")
	while goOn:
		selection = selections.pop()
		print_poem(selection)
		if len(selections) > 0:
			goOn = validate("\nWould you like another poem in this collection? ")
		else:
			goOn = False
	print("I hope you liked the poetry that I found for you!")

if __name__ == '__main__':
	main()