#!/usr/bin/env python
"""
A panflute filter for Pandoc JSON which looks for certain text sequences
beginning with `[_` and turns them into interactive questions with
check-answer and reveal-solution buttons. Extra formatting options may be
specified as letters before the initial closing `]`. A pair of `[|`
followed by `|]` marks the answer for the question, which can span
multiple lines and/or include additional markup. The recognized
patterns are:


## Fill-in-the-Blank

`[___][|answer|]`

This is a fill-in-the-blank. More than 3 underscores are allowed, but
they must be contiguous, with no spaces. By default it is
case-insensitive and ignores leading and trailing whitespace, plus each
run of internal whitespace is treated as a single space. This is still
pretty fragile though, so there are additional options:

- 'C' case-sensitive.
- 'w' don't trim or normalize whitespace.
- 'n' treat answer as a number. Round answer provided (and solution) to 5
    significant figures and then compare.
- 'N' like 'n' but rounds to 10 significant figures.
- 'i' treat answer as an integer. Compare against the provided answer
    exactly.
- 'u' last non-digit part of the correct answer is units, units must be
    provided in the answer and must match, but metric-conversion will be
    performed if 'n'/'N'/'i' is also given. Whitespace before the units
    is not necessary in the answer (as long as they're up against a
    number).

So for example,  `[___nu][300 mA] will accept answers including `300.001
mA`, `300mA`, `0.3 A`. Unit abbreviations are not understood, so you
can't answer `300 milli-amps`, it has to match the abbreviation (or
full-name) given in the correct answer.


## Short Answer

`[_...][|answer|]`

There must be exactly 1 underscore and 3 periods. Unlike
fill-in-the-blank, the answer is not checked automatically: there is no
"check answer" button and the reader has to compare against the provided
example answer themselves. A `textarea` element is provided instead of
an `input type=text` when possible.


## Multiple-Choice

```txt
[__][|Option 1|]
[_x][|Correct option 1|]
[__][|Option 2|]
[_x][|Correct option 2|]

[__][|Option 1|]
[_r][|Single correct option|]
[__][|Option 2|]

[_d][|Drop-down menu correct option|]
[__][|Option
whose text is split across multiple lines.

And which has multiple paragraphs.
|]
[__][|Drop-down menu incorrect option|]

[_c][|Toggle-on-click options|]
[__][|Incorrect toggle-on-click option|]
```

Just two underscores (or an underscore followed by an 'x') indicates an
option in a multiple-choice prompt. Stack multiple directly after each
other to create the full prompt. This will use checkboxes and allow for
selection of multiple answers. To allow for only single-answer selection
via radio buttons, use '_r' instead of '_x' for one or more correct
answers. With multiple correct options, only one must be chosen, unlike
'_x' where the answer is only correct if all correct options are chosen
simultaneously.

Other single-select forms are '_d' and '_c', which generate drop-down
menus and cycle-on-click text respectively.


## No-input Questions

`[_.][|answer|]`

This format will generate a "show answer" button without any input
elements. Useful when you want students to check their answer for some
complex task they can't easily upload results from.


## Hints, Feedback, and Explanations

```txt
How much current will flow through a circuit with 3 V potential across a
10 Ω resistor? Give your answer as a number with a unit symbol at the
end.

[___nu][|0.3 A|]

[_h][|Use Ohm's law.|]
[_f][|Ohm's law is $V = IR$. Solve it for current $I$ and then plug in
the values given.|]
[_f]/[.0-9]$/[|You didn't include units in your answer.|]

[_e][|
Ohm's law states that $V = IR$ which when solving for current ($I$)
yields:

$I = \\frac{V}{R}$

So for 3 volts and 10 ohms, we get 3 / 10 = 0.3 amperes (A), or
equivalently 300 milli-amps (mA).
|]
```

'[_h]', '[_f]', and '[_e'] "questions" attach themselves to the previous
normal question. '_h' adds a 'show hint' button which shows the given
hint. '_f' is feedback, to be displayed after 'check answer' is
selected. Feedback items may have a regular expression within slashes
and will only display if the pattern matches somewhere in the
(whitespace-normalized) answer, otherwise feedback items are only shown
if the answer is wrong. For question types which don't permit answer
checking, feedback is still shown when 'show answer' is selected and the
answer value matches the feedback's regular expression, or if it has
none, whenever 'show answer' is selected. Other question types do not
display feedback when 'show answer' is selected. '_e' is an explanation,
which is only shown when the correct answer is checked, or when the 'show
answer' option is selected.

'_f' can also be used to override the normal correctness-checking
mechanism: add 'c' to specify that answers matching the associated
regular expression should count as correct. 'n' and/or 'u' can be mixed
in here to provide the same number- and/or unit-checking behavior as a
normal short-answer question. This can be used to provide for
fill-in-the-blank questions with multiple correct answers (in this case,
the feedback text itself may be omitted):

```txt
What color is a stoplight? [___][|red|] [_fc]/^yellow$/i[||] [_fc]/^green$/i[||]

Name one prime number smaller than 10: [___][|2|]
[_fc]/^3$/[||]
[_fc]/^5$/[||]
[_fc]/^7$/[||]
[_fc]/^two$/i[||]
[_fc]/^three$/i[||]
[_fc]/^five$/i[||]
[_fc]/^seven$/i[||]
```

Note the use of '^' and '$' to ensure our test expressions match the
whole answer, and '/i' for ignore-case. By default the expression just
has to match somewhere in the answer.


## Units

When students answer fill-in-the-blank questions that have a 'unit'
part, (using the 'u' flag), their units are normalized to abbreviations
before being compared against the correct answer. So for example, an
answer of '3 square meters' should become '3 m²', and '2 kilograms per
s' should become '2 Kg/s' There are some built-in patterns for unit
conversion, but you can add a custom pattern by creating a "question"
with spec '_U'. The regex part of the question (see feedback, above)
specifies what to convert, and the answer part specifies what to convert
into. So for example, if you expect students to use light-years in their
answers, you could do:

```txt
[_U]/light.?years?/[|ly|]
```

After such a definition, answers like '3 light years', '3 light-years',
and '3 lightyear' would all be simplified to '3 ly'.

The resulting question and any empty parent elements will be removed from
the document by Javascript, which will add the specified rule to its set
of unit simplification rules. You may include backreferences in the
answer part if you have capture groups in the regex, these take the form
'$N' in Javascript, where N is a number counting from 1 for the first
group.

The default simplification rules include:

- 'square X' -> 'x²'
- 'cubic X' -> 'x³'
- SI unit prefixes -> their 1-letter versions
- SI base units: seconds, meters, grams, amperes, (degrees) kelvin,
    moles, and candelas.
- SI derived units: radian, steradian, hertz, newton, pascal, joule,
    watt, coulomb, volt, farad, ohm, siemens, weber, tesla, henry,
    (degrees) celsius, lumen, lux, becquerel, gray, sievert, and katal.

Note that although the system can convert the words 'degrees kelvin' (or
just 'kelvin') into the symbol 'K', it cannot make Celsius/Kelvin
conversions. (TODO: Get it to do that.)


## Groups

```txt
[_G]
Fill in the [___][|blanks|] in this [___][|sentence|].
[_E]
```

Questions between '[_G]' and '[_E]' markers will be grouped together and
get a single set of check-answer and show-answer buttons, although custom
hints and feedback for each individual question will be shown next to
that question. This also changes the show-answer behavior to indicate the
correct answer by changing the input value rather than by adding a popup.

Groups cannot be nested (inner groups will be ignored).

Use this for situations like lots of options in a table where having a
button for each would be too cluttered.

## Version history

- v1.0: Initial release version; adds _fc alternate-correct-answers
    functionality and fixes some minor bugs. Changes button text back to
    letters/symbols.
- v0.2: Start of version numbering. Changes emoticons to icons. -.-
"""

__version__ = 1.0

from typing import List, Dict, Any

import copy
import sys

import panflute as pf  # type: ignore


# CSS for discovery elements
DISCOVERY_CSS = '''
.question button {
  font-size: 14pt;
  margin: 2pt 4pt;
}

.question textarea {
  width: 80%;
}

.question .answer,
.question .explanation,
.question .hint,
.question .feedback,
.question .condole,
.question .congrats {
    border-width: 1pt;
    border-style: solid;
    border-radius: 2pt;
    padding: 2pt;
    margin: 3pt;
}

.question .answer {
    display: none;
    background: #fef;
    color: #402;
    border-color: #faf;
}

.question.show .answer {
    display: block;
}

.question .explanation {
    display: none;
    background: #edf;
    color: #204;
    border-color: #eaf;
}

.question.show .explanation, .question.show-correct .explanation {
    display: block;
}

.question .hint {
    display: none;
    background: #ffe;
    color: #330;
    border-color: #fea;
}

.question.show-hints .hint {
    display: block;
}

.question .feedback {
    display: none;
    background: #fed;
    color: #420;
    border-color: #fba;
}

.question .feedback.show {
    display: block;
}

.question .condole {
    display: none;
    background: #fcb;
    color: #300;
    border-color: #f66;
}

.question .congrats {
    display: none;
    background: #bcf;
    color: #003;
    border-color: #66f;
}

.question.show-incorrect .condole {
    display: block;
}

.question.show-correct .congrats {
    display: block;
}

.question .select-option {
    padding: 0pt 2pt;
    margin: 0pt 2pt;
    line-height: 140%;
}

.question.show .select-option.correct {
    border: 2pt solid #faf;
    border-radius: 2pt;
}

.cycler {
    color: #029;
    text-decoration: underline;
    cursor: pointer;
}
'''

# Javascript code to run the interactivity of the questions
DISCOVERY_JS = '''

// Set to true to enable logs for answer saving/loading
var DEBUG_STORAGE = false;

// Text for various buttons
const SHOW_HINT_TEXT = "H";
const HIDE_HINT_TEXT = "H";
const CHECK_ANSWER_TEXT = "✓";
const SHOW_ANSWER_TEXT = "?";
const HIDE_ANSWER_TEXT = "?";

function testTrue(val, name) {
    console.assert(val, name + " is wrong");
    return val;
}


function testVal(got, exp, name) {
    console.assert(
        got == exp,
        name + " is wrong: got " + got + " but expected " + exp
    );
    return got == exp;
}


// SI prefixes ('da' makes us sad; we put it BEFORE 'd'):
let SI_PREFIXES = [
    'Q', 'R', 'Y', 'Z', 'E', 'P', 'T', 'G', 'M', 'K', 'h', 'da',
    'd', 'c', 'm', 'μ', 'n', 'p', 'f', 'a', 'z', 'y', 'r', 'q'
];

// Exponents for each unit
let SI_EXPONENTS = {
    'Q': 30,
    'R': 27,
    'Y': 24,
    'Z': 21,
    'E': 18,
    'P': 15,
    'T': 12,
    'G': 9,
    'M': 6,
    'K': 3,
    'h': 2,
    'da': 1,
    'd': -1,
    'c': -2,
    'm': -3,
    'μ': -6,
    'n': -9,
    'p': -12,
    'f': -15,
    'a': -18,
    'z': -21,
    'y': -24,
    'r': -27,
    'q': -30
}


// Unit conversion rules
// (Note extra backslashes in Python source because we're in a Python
// string)
let UNIT_CONVERSIONS = [
    [/square (\\w+)/g, "$1²"],
    [/cubic (\\w+)/g, "$1³"],
    [/\\bper\\b/g, "/"], // divisor
    // SI prefix long forms
    [/quetta-?/ig, "Q"], [/ronna-?/ig, "R"], [/yotta-?/ig, "Y"],
    [/zetta-?/ig, "Z"], [/exa-?/ig, "E"], [/peta-?/ig, "P"],
    [/tera-?/ig, "T"], [/giga-?/ig, "G"], [/mega-?/ig, "M"],
    [/kilo-?/ig, "K"], [/hecto-?/ig, "h"], [/deca-?/ig, "da"],
    [/deci-?/ig, "d"], [/centi-?/ig, "c"], [/milli-?/ig, "m"],
    [/micro-?/ig, "μ"], [/nano-?/ig, "n"], [/pico-?/ig, "p"],
    [/femto-?/ig, "f"], [/atto-?/ig, "a"], [/zepto-?/ig, "z"],
    [/yocto-?/ig, "y"], [/ronto-?/ig, "r"], [/quecto-?/ig, "q"],
    // SI base units
    [/seconds?/ig, "s"], [/(metre)|(meter)s?/ig, "m"], [/grams?/ig, "g"],
    [/amperes?/ig, "A"], [/amps?/ig, "A"], [/(degrees?)? kelvin/ig, "°K"],
    [/moles?/ig, "mol"], [/candelas?/ig, "cd"],
    // coherent SI derived units
    [/radians?/ig, "rad"], [/steradians?/ig, "sr"], [/hertz/ig, "Hz"],
    [/newtons?/ig, "N"], [/pascals?/ig, "Pa"], [/joules?/ig, "J"],
    [/watts?/ig, "W"], [/coulombs?/ig, "C"], [/volts?/ig, "V"],
    [/farads?/ig, "F"], [/ohms?/ig, "Ω"], [/siemens/ig, "S"],
    [/webers?/ig, "Wb"], [/teslas?/ig, "T"], [/henry|(ies)/ig, "H"],
    [/(degrees?)? celsius/ig, "°C"], [/lumens?/ig, "lm"], [/lux/ig, "lx"],
    [/becquerels?/ig, "Bq"], [/grays?/ig, "Gy"], [/sieverts?/ig, "Sv"],
    [/katals?/ig, "kat"]
];

// Expressions of derived SI units in terms of base units
let UNIT_EQUIVALENCES = {
    'Hz': '/s',
    'N': 'Kg m/s²',
    'Pa': 'Kg/m s²',
    'J': 'Kg m²/s²',
    'W': 'Kg m²/s³',
    'C': 's A',
    'V': 'Kg m²/s³ A',
    'F': 's⁴ A²/Kg m²',
    'Ω': 'Kg m²/s³ A²',
    'S': 's³ A²/Kg m²',
    'Wb': 'Kg m²/s² A',
    'T': 'Kg/s² A',
    'H': 'Kg m²/s² A²',
    // No base °C -> °K rule because that would be wrong when the unit
    // was alone.
    'lm': 'cd sr', // steradians are included to differentiate from candelas
    'lx': 'cd sr/m²',
    'Bq': '/s',
    'Gy': 'm²/s²',
    'Sv': 'm²/s²',
    'kat': 'mol/s',
};

// Regex for splitting units
let UNITS_SPLIT = /[\\s-·]+/g;


// Changes fancy single & double quotes into ambidextrous versions since
// that's likely how input will come in.
function straightenQuotes(string) {
    return string.replace(/[“”]/g, '"').replace(/[‘’]/g, "'");
}


// Returns the unit with any trailing exponent digits stripped off.
function withoutExponent(unit) {
    return unit.replace(/[¹²³⁴⁵⁶⁷⁸⁹⁰⁻]+$/, '');
}


// Scans superscript numerals at the end of a string, returning the
// total exponent as an integer. Returns 1 when no superscript numerals
// are found.
function exponentValue(unit) {
    let match = /[¹²³⁴⁵⁶⁷⁸⁹⁰⁻]+$/.exec(unit);
    if (match == null) {
        return 1;
    } else {
        let digits = '';
        for (let char of match[0]) {
            if (char == '⁻') {
                digits += '-';
            } else if (char == '⁰') {
                digits += '0';
            } else if (char == '¹') {
                digits += '1';
            } else if (char == '²') {
                digits += '2';
            } else if (char == '³') {
                digits += '3';
            } else if (char == '⁴') {
                digits += '4';
            } else if (char == '⁵') {
                digits += '5';
            } else if (char == '⁶') {
                digits += '6';
            } else if (char == '⁷') {
                digits += '7';
            } else if (char == '⁸') {
                digits += '8';
            } else if (char == '⁹') {
                digits += '9';
            }
        }
        return parseInt(digits);
    }
}


// Converts an integer into exponent symbols.
function exponentStr(num) {
    let digits = num.toString();
    let result = '';
    for (let d of digits) {
        if (d == '-') {
            result += '⁻';
        } else if (d == '0') {
            result += '⁰';
        } else if (d == '1') {
            result += '¹';
        } else if (d == '2') {
            result += '²';
        } else if (d == '3') {
            result += '³';
        } else if (d == '4') {
            result += '⁴';
        } else if (d == '5') {
            result += '⁵';
        } else if (d == '6') {
            result += '⁶';
        } else if (d == '7') {
            result += '⁷';
        } else if (d == '8') {
            result += '⁸';
        } else if (d == '9') {
            result += '⁹';
        }
    }
    return result;
}

// Given a single-unit string, returns a base-units string with any SI
// prefix stripped off, along with an exponent that represents the
// prefix that was stripped. For example, 'Kg' will return ['g', 3], or
// 'μlx' will return ['lx', -6]. The SI prefix must be at the start of
// the string, and must be followed by at least one letter. Unit
// exponents at the end of the string will be accounted for: 'Kg³' would
// return ['g', 9]. If a known SI prefix is not found, the original unit
// string is returned unchanged, with 0 as the exponent.
function deprefix(unit) {
    result = withoutExponent(unit);
    let unitExp = exponentValue(unit);
    let exp = 0;
    for (let pre of SI_PREFIXES) {
        if (result.length > pre.length && result.startsWith(pre)) {
            result = result.substring(pre.length);
            exp = SI_EXPONENTS[pre];
            break;
        }
    }
    exp *= unitExp;
    if (unitExp != 1) {
        return [result + exponentStr(unitExp), exp];
    } else {
        return [result, exp];
    }
}


// Tests for deprefix
function testDeprefix() {
    let pass = true;

    let r0 = deprefix('g');
    if (
        !testVal(r0.length, 2, "Deprefix 0 length")
     || !testVal(r0[0], 'g', "Deprefix 0 unit")
     || !testVal(r0[1], 0, "Deprefix 0 exponent")
    ) {
        console.error("Bad deprefix 0 result:");
        console.error(r0);
        pass = false;
    }

    let r1 = deprefix('Kg');
    if (
        !testVal(r1.length, 2, "Deprefix 1 length")
     || !testVal(r1[0], 'g', "Deprefix 1 unit")
     || !testVal(r1[1], 3, "Deprefix 1 exponent")
    ) {
        console.error("Bad deprefix 1 result:");
        console.error(r1);
        pass = false;
    }

    let r2 = deprefix('μlx');
    if (
        !testVal(r2.length, 2, "Deprefix 2 length")
     || !testVal(r2[0], 'lx', "Deprefix 2 unit")
     || !testVal(r2[1], -6, "Deprefix 2 exponent")
    ) {
        console.error("Bad deprefix 2 result:");
        console.error(r2);
        pass = false;
    }

    let r3 = deprefix('Kg³');
    if (
        !testVal(r3.length, 2, "Deprefix 3 length")
     || !testVal(r3[0], 'g³', "Deprefix 3 unit")
     || !testVal(r3[1], 9, "Deprefix 3 exponent")
    ) {
        console.error("Bad deprefix 3 result:");
        console.error(r3);
        pass = false;
    }

    let r4 = deprefix('m⁴');
    if (
        !testVal(r4.length, 2, "Deprefix 4 length")
     || !testVal(r4[0], 'm⁴', "Deprefix 4 unit")
     || !testVal(r4[1], 0, "Deprefix 4 exponent")
    ) {
        console.error("Bad deprefix 4 result:");
        console.error(r4);
        pass = false;
    }

    if (pass) {
        console.warn("deprefix tests passed.");
    }
}


// Given a unit list and a new unit being added, mixes in the new unit,
// cancelling and/or adding exponents if another copy of that unit
// already exists in the list.
function mixInUnit(unitList, newUnit) {
    let trimmed = withoutExponent(newUnit);
    let unitExp = exponentValue(newUnit);
    for (let i = 0; i < unitList.length; ++i) {
        let found = unitList[i];
        if (trimmed == withoutExponent(found)) {
            let newExp = exponentValue(found) + unitExp;
            if (newExp == 0) {
                unitList.splice(i, 1);
            } else if (newExp == 1) {
                unitList[i] = trimmed; // no '1' exponent
            } else {
                unitList[i] = trimmed + exponentStr(newExp);
            }
            return;  // found a match: don't continue
        }
    }

    // If we didn't find a match
    unitList.push(newUnit);
}


// Given a units string, standardizes it as much as possible into base
// SI units without prefixes. Returns an array containing:
// - A sub-array of strings containing units.
// - An integer specifying the exponent for the power of ten that can
//   multiply a provided number in the given units to express it in the
//   result units.
//
// For example, if the string is 'kilo-Newtons' the result would be:
//
// [ ['g', 'm', 's⁻²'], 6 ]
//
// (Note how the 'kg' part of Newtons becomes 'g' with an extra +3
// exponent.)
//
// Similarly, if the string is 'mm/s' the result would be:
//
// [ ['m', 's⁻¹'], -6 ]
//
// Returns 'undefined' if it encounters a malformed unit string.
function standardizeUnits(unitString) {
    // Apply unit conversions:
    for (let [rec, sub] of UNIT_CONVERSIONS) {
        unitString = unitString.replace(rec, sub);
    }
    // Figure out numerator and denominator units
    let split = unitString.split('/');
    let numeratorUnits = [];
    let denominatorUnits = [];
    if (split.length == 1) {
        numeratorUnits = split[0].split(UNITS_SPLIT)
    } else if (split.length == 2) {
        numeratorUnits = split[0].split(UNITS_SPLIT);
        denominatorUnits = split[1].split(UNITS_SPLIT);
    } else {
        return undefined;
    }
    numeratorUnits = numeratorUnits.map(
        (x) => x.trim()
    ).filter((x) => x.length > 0);
    denominatorUnits = denominatorUnits.map(
        (x) => x.trim()
    ).filter((x) => x.length > 0);

    // First stage of cleaning
    let finalExp = 0;
    let cleanUnits = [];
    for (let nU of numeratorUnits) {
        let [base, exp] = deprefix(nU);
        finalExp += exp;
        cleanUnits.push(base);
    }
    for (let dU of denominatorUnits) {
        let [base, exp] = deprefix(dU);
        finalExp -= exp;
        let baseExp = exponentValue(base);
        cleanUnits.push(withoutExponent(base) + exponentStr(-baseExp));
    }

    // Now we apply equivalences:
    let finalUnits = [];
    for (let cU of cleanUnits) {
        let uBase = withoutExponent(cU);
        let uExp = exponentValue(cU);
        if (UNIT_EQUIVALENCES.hasOwnProperty(uBase)) {
            let [eUnits, eExp] = standardizeUnits(UNIT_EQUIVALENCES[uBase]);
            finalExp += eExp * uExp;
            for (let eU of eUnits) {
                let eUnit = withoutExponent(eU);
                let eExp = exponentValue(eU);
                let cExp = eExp * uExp;
                if (cExp == 0) {  // shouldn't be possible
                    console.warn("ERROR: Exponent 0 in cancellation.");
                } else if (cExp == 1) {
                    mixInUnit(finalUnits, eUnit);
                } else {
                    mixInUnit(finalUnits, eUnit + exponentStr(cExp));
                }
            }
        } else {
            mixInUnit(finalUnits, cU);
        }
    }

    return [finalUnits, finalExp];
}


// Unit tests for standardizeUnits function.
function testStandardizeUnits() {
    let pass = true;

    let r0 = standardizeUnits('m');
    if (
        !testVal(r0.length, 2, "Standardized units 0 length")
     || !testVal(r0[0].length, 1, "Standardized units 0 length")
     || !testVal(r0[1], 0, "Standardized exponent 0")
     || !testVal(r0[0][0], 'm', "Standardized unit 0")
    ) {
        console.error("Result 0 is:");
        console.error(r0);
        pass = false;
    }

    let r1 = standardizeUnits('kilo-Newtons');
    if (
        !testVal(r1.length, 2, "Standardized units 1 length")
     || !testVal(r1[0].length, 3, "Standardized units 1 length")
     || !testVal(r1[1], 6, "Standardized exponent 1")
     || !testVal(r1[0][0], 'g', "Standardized unit 1 part 1")
     || !testVal(r1[0][1], 'm', "Standardized unit 1 part 2")
     || !testVal(r1[0][2], 's⁻²', "Standardized unit 1 part 3")
    ) {
        console.error("Result 1 is:");
        console.error(r1);
        pass = false;
    }

    let r2 = standardizeUnits('cubic centimeters per joule');
    if (
        !testVal(r2.length, 2, "Standardized units length 2")
     || !testVal(r2[0].length, 3, "Standardized units length 2")
     || !testVal(r2[1], -9, "Standardized exponent 2")
     || !testVal(r2[0][0], 'm', "Standardized unit 2 part 1")
     || !testVal(r2[0][1], 'g⁻¹', "Standardized unit 2 part 2")
     || !testVal(r2[0][2], 's²', "Standardized unit 2 part 3")
    ) {
        console.error("Result 2 is:");
        console.error(r2);
        pass = false;
    }

    let r3 = standardizeUnits('cm³ / J');
    if (
        !testVal(r3.length, 2, "Standardized units length 2")
     || !testVal(r3[0].length, 3, "Standardized units length 2")
     || !testVal(r3[1], -9, "Standardized exponent 2")
     || !testVal(r3[0][0], 'm', "Standardized unit 2 part 1")
     || !testVal(r3[0][1], 'g⁻¹', "Standardized unit 2 part 2")
     || !testVal(r3[0][2], 's²', "Standardized unit 2 part 3")
    ) {
        console.error("Result 3 is:");
        console.error(r3);
        pass = false;
    }

    let r4 = standardizeUnits('s·GHz');
    if (
        !testVal(r4.length, 2, "Standardized units length 4")
     || !testVal(r4[0].length, 0, "Standardized units length 4")
     || !testVal(r4[1], 9, "Standardized exponent 4")
    ) {
        console.error("Result 4 is:");
        console.error(r4);
        pass = false;
    }

    if (pass) {
        console.warn("standardizeUnits tests passed.");
    }
}


// Compares answers and returns true if the provided answer is correct
// and false otherwise. The flags control exact comparison behavior.
function compareAnswer(provided, correct, flags) {
    // Normalize whitespace
    if (!flags.includes('w')) {
        provided = provided.trim();
        correct = correct.trim();
        provided = provided.split(/\\s+/).join(' ');
        correct = correct.split(/\\s+/).join(' ');
    }

    // Normalize quotation marks
    provided = straightenQuotes(provided);
    correct = straightenQuotes(correct);

    // Strip off unit to compare separately
    let prUnit = '';
    let crUnit = '';
    if (flags.includes('u')) {
        prUnit = /[^0-9.]+$/.exec(provided);
        if (prUnit === null) {
            prUnit = '';
        } else {
            prUnit = prUnit[0];
        }
        provided = provided.substring(0, provided.length - prUnit.length);
        crUnit = /[^0-9.]+$/.exec(correct);
        if (crUnit === null) {
            crUnit = '';
        } else {
            crUnit = crUnit[0];
        }
        correct = correct.substring(0, correct.length - crUnit.length);
    }

    // Normalize case
    if (!flags.includes('C')) {
        provided = provided.toLocaleLowerCase();
        correct = correct.toLocaleLowerCase();
    }

    // TODO: Mechanism for NaN feedback
    if (flags.includes('n') || flags.includes('N') || flags.includes('i')) {
        let prNum, crNum;
        if (flags.includes('i')) {
            prNum = parseInt(provided);
            crNum = parseInt(correct);
        } else {
            prNum = parseFloat(provided);
            crNum = parseFloat(correct);
        }
        // NaN is never the correct answer for a numerical question
        if (isNaN(crNum)) {
            console.warn(
                "Correct answer for numerical question was NOT parseable"
              + " as a number: '" + correct + "'!"
            );
            return false;
        }
        let prUnitStd = standardizeUnits(prUnit);
        let crUnitStd = standardizeUnits(crUnit);
        if (crUnitStd === undefined) {
            console.warn(
                "Correct answer for numerical question has invalid"
              + " units: '" + correct + "'!"
            );
            return false;
        }
        if (prUnitStd === undefined) {
            // TODO: Invalid units warning here!
            return false;
        }
        let [prUnits, prConvExp] = prUnitStd;
        let [crUnits, crConvExp] = crUnitStd;
        if (prUnits.length != crUnits.length) {
            // TODO: Units mismatch warning here
            return false;
        }
        prUnits.sort();
        crUnits.sort();
        for (let i = 0; i < prUnits.length; ++i) {
            if (prUnits[i] != crUnits[i]) {
                // Units mismatch warning here
                return false;
            }
        }
        // Now we know that the units match; need to adjust number
        let adjustExp = prConvExp - crConvExp;
        prNum = prNum * (10 ** adjustExp);

        // Round to desired precision & compare
        if (flags.includes('i')) {
            return prNum == crNum;
        } else {
            let precision = 5;
            if (flags.includes('N')) {
                precision = 10;
            }
            prNum = prNum.toPrecision(precision);
            crNum = crNum.toPrecision(precision);
            return prNum == crNum;
            // TODO: You're-close hint!
        }
    } else {
        return provided == correct && prUnit == crUnit;
    }
}


function testCompareAnswer() {
    if (
        testTrue(compareAnswer('5 mA', '5 mA', 'nu'), '5 mA')
     && testTrue(compareAnswer('5 mA', '0.005 A', 'nu'), '5 mA 2')
     && testTrue(compareAnswer('5 mA', '0.000005 MC mHz', 'nu'), '5 mA 3')
     && testTrue(compareAnswer('0.000005 MC mHz', '5 mA', 'nu'), '5 mA 4')
    ) {
        console.warn("compareAnswer tests passed.");
    }
}

// Given a question element, builds the regular expression for that
// question from its data-regex and data-regex-flags attributes and
// returns a RegExp object. Returns null if the given element doesn't
// have a regular expression attached.
function buildQuestionRegex(fbElem) {
    let regex = fbElem.getAttribute('data-regex');
    if (regex === null) {
        return null;
    } else {
        let flags = fbElem.getAttribute('data-regex-flags');
        return new RegExp(regex, flags);
    }
}

// Given a question element, returns a 2-element array containing the
// current answer value and a boolean indicating whether or not it's
// correct. Returns 'undefined' as the second element for non-checkable
// question types.
function extractAnswer(qElem) {
    let qType = getQuestionType(qElem);

    let answer = '';  // to be tested against hint regexes
    let correct = undefined;

    if (qType == 'blank') {
        answer = qElem.querySelector('input').value;
        let against = qElem.querySelector('.answer-value').innerText;
        let flags = qElem.getAttribute('data-flags');
        correct = compareAnswer(answer, against, flags);
    } else if (qType == 'option') {
        let flavor = qElem.getAttribute('data-select-flavor');
        correct = true;
        if (flavor == "dropdown") {
            let menu = qElem.querySelector('select');
            answer = "";
            for (let opt of menu.options) {
                let on = opt.selected;
                if (on) {
                    if (answer.length > 0) {
                        answer += ';;';
                    }
                    answer += opt.value;
                }
                if (
                    (!on && opt.classList.contains('correct'))
                 || (on && !opt.classList.contains('correct'))
                ) {
                    correct = false;
                }
            }
        } else if (flavor == "cycle") {
            for (let opt of qElem.querySelectorAll('.select-option')) {
                let on = opt.style.display != 'none';
                if (on) {
                    if (answer.length > 0) {
                        answer += ';;';
                    }
                    answer += opt.innerText;
                }
                if (
                    (!on && opt.classList.contains('correct'))
                 || (on && !opt.classList.contains('correct'))
                ) {
                    correct = false;
                }
            }
        } else {  // Same procedure for 'multiple' or 'radio'
            for (let opt of qElem.querySelectorAll('.select-option')) {
                let checkbox = opt.querySelector('input');
                if (checkbox === null) {
                    console.warn("Missing input element for option question:");
                    console.warn(qElem);
                    answer = '';
                    correct = undefined;
                } else {
                    let on = checkbox.checked;
                    if (on) {
                        if (answer.length > 0) {
                            answer += ';;';
                        }
                        answer += opt.innerText;
                    }
                    if (
                        (!on && opt.classList.contains('correct'))
                     || (on && !opt.classList.contains('correct'))
                    ) {
                        correct = false;
                    }
                }
            }
        }
    } else if (qType == 'short-answer') {
        let inp = qElem.querySelector('textarea');
        if (inp == null) {
            inp = qElem.querySelector('input');
        }
        answer = inp.value;
        correct = undefined;
    } else if (
        qType == 'no-answer'
     || qType == 'GROUP_START'
     || qType == 'GROUP_END'
    ) {
        answer = '';
        correct = undefined;
    } else {
        console.warn("Can't get answer for question type: '" + qType + "'.");
        answer = '';
        correct = undefined;
    }

    // Check for _fc feedback items and mark answer as correct if any
    // match
    for (let fb of qElem.querySelectorAll('.feedback')) {
        let flags = fb.getAttribute('data-flags');
        if (flags.includes('c')) {
            // A 'correct-answer' feedback item
            let regex = buildQuestionRegex(fb);
            if (regex === null || regex.exec(answer)) {
                correct = true;
            }
        }
    }
    return [answer, correct];
}


function checkAnswer() {
    let qElem = this.parentElement;

    // Save the current answer value
    saveAnswer(qElem);

    let [answer, correct] = extractAnswer(qElem);

    // display appropriate feedback items
    showFeedback(qElem, answer, correct);

    if (correct === undefined) {
        let subs = qElem.getAttribute('data-sub-ids');
        if (subs) {
            correct = true;
            for (let subID of subs.split(' ')) {
                if (subID) {
                    let sub = document.getElementById(subID);
                    let [subAns, subCorrect] = extractAnswer(sub);
                    if (!subCorrect) {
                        correct = false;
                    }
                }
            }
        } else {
            console.warn(
                "Can't check answer for question type: '"
              + getQuestionType(qElem)
              + "'; toggling answer visibility instead."
            );
            toggleAnswer.call(this);
            return;
        }
    }

    if (!correct) {  // got it wrong
        // display matching feedback items
        qElem.classList.add('show-incorrect');
        qElem.classList.remove('show-correct');

        let subs = qElem.getAttribute('data-sub-ids');
        if (subs) {
            for (let subID of subs.split(' ')) {
                if (subID) {
                    let sub = document.getElementById(subID);
                    let [subAns, subCorrect] = extractAnswer(sub);
                    showFeedback(sub, subAns, subCorrect);
                    sub.classList.remove('show-correct');
                }
            }
        }
    } else {  // got it right
        // display explanation(s); hide feedback
        qElem.classList.add('show-correct');
        qElem.classList.remove('show-incorrect');

        let subs = qElem.getAttribute('data-sub-ids');
        if (subs) {
            for (let subID of subs.split(' ')) {
                if (subID) {
                    let sub = document.getElementById(subID);
                    let [subAns, subCorrect] = extractAnswer(sub);
                    hideFeedback(sub, subAns, subCorrect);
                    sub.classList.add('show-correct');
                }
            }
        }
    }
}


// Shows relevant feedback elements based on the current answer value
function showFeedback(qElem, answer, correct) {
    hideFeedback(qElem);  // first hide all
    if (answer === undefined) {
        [answer, correct] = extractAnswer(qElem);
    }
    for (let fb of qElem.querySelectorAll('.feedback')) {
        let flags = fb.getAttribute('data-flags');
        let regex = buildQuestionRegex(fb);
        if (
            correct == flags.includes('c')
         && (regex === null || regex.exec(answer))
         && qElem.querySelector('.answer-value').innerText.trim() != ''
             // answer value must be non-empty to be worth showing
        ) {
            fb.classList.add('show');
        }
    }
}


// Hides all feedback elements
function hideFeedback(qElem) {
    for (let fb of qElem.querySelectorAll('.feedback')) {
        fb.classList.remove('show');
    }
}


// Sets the input value to the correct answer
function setToAnswer(qElem) {
    let qType = getQuestionType(qElem);
    if (qType == 'blank' || qType == 'no-answer') {
        let correct = qElem.querySelector('.answer-value').innerText;
        qElem.querySelector('input').value = correct;
    } else if (qType == 'option') {
        let flavor = qElem.getAttribute('data-select-flavor');
        if (flavor == 'dropdown') {
            let dropdown = qElem.querySelector('select');
            for (let opt of dropdown.options) {
                if (opt.classList.contains('correct')) {
                    opt.selected = true;
                }
            }
        } else if (flavor == 'cycle') {
            let cycler = qElem.querySelector('.cycler');
            for (let opt of cycler.querySelectorAll('.select-option')) {
                if (opt.classList.contains('correct')) {
                    opt.style.display = "inline";
                    opt.setAttribute('aria-selected', true);
                } else {
                    opt.style.display = "none";
                    opt.setAttribute('aria-selected', false);
                }
            }
        } else {  // flavor is 'radio' or 'multiple'
            for (let opt of qElem.querySelectorAll('input')) {
                if (opt.classList.contains('correct')) {
                    opt.checked = true;
                } else {
                    opt.checked = false;
                }
            }
        }
    } else if (qType == 'short-answer') {
        let correct = qElem.querySelector('.answer-value').innerText;
        let input = qElem.querySelector('textarea');
        input.value = correct;
    } else {
        console.warn("Can't set answer for question type: '" + qType + "'.");
    }
}


// Extracts the correct answer value as a string (joining multiple
// correct answers with double semicolons where relevant). Returns empty
// string as a default if a question doesn't have a correct answer.
function getCorrectAnswer(qElem) {
    let qType = getQuestionType(qElem);
    if (qType == 'blank' || qType == 'no-answer') {
        return qElem.querySelector('.answer-value').innerText;
    } else if (qType == 'option') {
        let flavor = qElem.getAttribute('data-select-flavor');
        if (flavor == 'dropdown') {
            let dropdown = qElem.querySelector('select');
            let result = "";
            for (let opt of dropdown.options) {
                if (opt.classList.contains('correct')) {
                    if (result == "") {
                        result += opt.value;
                    } else {
                        result += ";;" + opt.value;
                    }
                }
            }
            return result;
        } else if (flavor == 'cycle') {
            let cycler = qElem.querySelector('.cycler');
            for (let opt of cycler.querySelectorAll('.select-option')) {
                if (opt.classList.contains('correct')) {
                    return opt.innerText;
                }
            }
        } else {  // flavor is 'radio' or 'multiple'
            let result = "";
            for (let opt of qElem.querySelectorAll('input')) {
                if (opt.classList.contains('correct')) {
                    if (result == "") {
                        result = opt.parentNode.innerText;
                    } else {
                        result += ";;" + opt.parentNode.innerText;
                    }
                }
            }
            return result;
        }
    } else if (qType == 'short-answer') {
        return qElem.querySelector('.answer-value').innerText;
    }
    // default value is empty string
    return "";
}


// Erases/resets the input value
function resetAnswer(qElem) {
    let qType = getQuestionType(qElem);
    if (qType == 'blank') {
        qElem.querySelector('input').value = '';
    } else if (qType == 'option') {
        let flavor = qElem.getAttribute('data-select-flavor');
        if (flavor == 'dropdown') {
            let dropdown = qElem.querySelector('select');
            for (let opt of dropdown.querySelectorAll('option')) {
                dropdown.value = opt.value;
                break;
            }
        } else if (flavor == 'cycle') {
            let cycler = qElem.querySelector('.cycler');
            let first = true;
            for (let opt of cycler.querySelectorAll('.select-option')) {
                if (first) {
                    opt.style.display = "inline";
                    opt.setAttribute('aria-selected', true);
                    first = false;
                } else {
                    opt.style.display = "none";
                    opt.setAttribute('aria-selected', false);
                }
            }
        } else {  // flavor is 'radio' or 'multiple'
            for (let opt of qElem.querySelectorAll('input')) {
                opt.checked = false;
            }
        }
    } else if (qType == 'short-answer') {
        let input = qElem.querySelector('textarea');
        input.value = '';
    } else if (qType == 'no-answer') {
        // nothing to do
    } else {
        console.warn("Can't reset answer for question type: '" + qType + "'.");
        answer = '';
        correct = undefined;
    }
}


// Given a question element, saves the current answer value to local
// storage, in usch a way that restoreSavedAnswer can try to restore it,
// even when the exact answer, question order, and/or question context may
// have changed (if too many things change, this won't always be
// possible).
function saveAnswer(qElem) {
    let fullKey = getQuestionKey(qElem);
    let qType = getQuestionType(qElem);
    let qID = getQID(qElem);
    QID__AKEY.set(qID, fullKey);
    AKEY__QID.set(fullKey, qID);

    // Clear any auto-save timeout that might be waiting
    let timeoutID = qElem.autoSaveTimeout;
    if (timeoutID != null) {
        window.clearTimeout(timeoutID);
    }
    qElem.autoSaveTimeout = null;

    if (DEBUG_STORAGE) {
        console.log("Save Answer for key: " + fullKey);
    }
    // Store current answer value under full key
    if (
        qType != "GROUP_END"
     && qType != "GROUP_START"
     && qType != "no-answer"
    ) {
        let answer = extractAnswer(qElem)[0];
        if (DEBUG_STORAGE) {
            console.log("Saving Answer: '" + answer + "'");
        }
        let storage = window.localStorage;
        storage.setItem(fullKey, answer);

        // Get list of keys by question type
        let urlPath = new URL(document.URL).pathname;
        let typeKey = urlPath + "#" + qType
        let byType = storage.getItem(typeKey);
        byType = insertOrUpdateAnswerKeyByType(byType, fullKey);
        storage.setItem(typeKey, byType);
    }

    // Save any sub-answers
    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                saveAnswer(document.getElementById(subID));
            }
        }
    }
}


// Given a string representing all question keys for a particular
// question type, and a possibly-new question key for an answer we just
// stored, return an updated string that incorporates that key, possibly
// replacing an old value stored for the same question ID.
function insertOrUpdateAnswerKeyByType(byTypeList, fullKey) {
	if (byTypeList == null || byTypeList == "") {
        return fullKey;
	}
    let parts = byTypeList.split(';;');
    let thisID = parseInt(
        fullKey.slice(fullKey.indexOf('#') + 1, fullKey.indexOf(':'))
    );
    if (isNaN(thisID)) {
      console.warn(
          "Invalid new qID: " + fullKey.slice(0, fullKey.indexOf(':'))
      );
    }
    let result = "";
    let matched = false;
    for (let key of parts) {
        let idPart = key.slice(key.indexOf('#') + 1, key.indexOf(':'));
        let qID = parseInt(idPart);
        if (isNaN(qID)) {
          console.warn("Invalid stored qID: " + idPart);
        }
        if (matched) {
            result += key + ';;';
        } else {
            if (qID == thisID) {
                result += fullKey + ';;';
                matched = true;
            } else if (qID > thisID) {
                result += fullKey + ';;' + key + ';;';
                matched = true;
            } else {
                result += key + ';;'
            }
        }
    }
    return result.slice(0, -2);
}

// Graft on Set.intersection if necessary
if (Set.prototype.intersection == null) {
    Set.prototype.intersection = function(other) {
        let result = new Set();
		for (elem of this) {
            if (other.has(elem)) {
                result.add(elem);
            }
		}
        return result;
	};
}


// Searches for a stored answer value for a particular question, and
// returns the stored answer string, or an empty string if it can't find
// a good match. Relies on the QID__AKEY mapping created during setup.
function retrieveAnswer(qElem) {
    let qID = getQID(qElem);
    key = QID__AKEY.get(qID);
    if (key == null) {
        return "";
    } else {
        return window.localStorage.getItem(key);
    }
}


// Searches for a stored answer key for a particular question, and
// returns the best matching key, or null if it can't find a good match.
// alreadyMatched should be a set-like object with a 'has' method;
// elements in it will be skipped over when trying to find a match.
// If perfectOnly is set to true, only perfect matches will be returned.
function findAnswerKey(qElem, alreadyMatched, perfectOnly=false) {
    let [qID, qType, ansID, ctxBefore, ctxAfter] = getIdentity(qElem);
    let fullKey = getQuestionKey(qElem);
    let storage = window.localStorage;
    let perfect = storage.getItem(fullKey);
    if (perfect != null) {
        if (alreadyMatched.has(fullKey)) {
            return null;
        } else {
            return fullKey;
        }
    }

    if (perfectOnly) {
        return null;
    }

    // No perfect match, let's look for a close match based on
    // before/after context.
    let urlPath = new URL(document.URL).pathname;
    let typeKey = urlPath + "#" + qType
    let sameType = storage.getItem(typeKey);
    if (sameType == null) {
        return null;
    }
    let ansMatches = new Set();
    let beforeMatches = new Set();
    let afterMatches = new Set();
    for (let piece of sameType.split(';;')) {
        let [kPath, kPieces] = piece.split('#');
        let [kID, kType, kAns, kBefore, kAfter] = kPieces.split(':');
        if (kPath != urlPath) {
            continue;
        }
        if (kAns == ansID && !alreadyMatched.has(piece)) {
            ansMatches.add(piece);
        }
        if (kBefore == ctxBefore && !alreadyMatched.has(piece)) {
            beforeMatches.add(piece);
        }
        if (kAfter == ctxAfter && !alreadyMatched.has(piece)) {
            afterMatches.add(piece);
        }
    }

    let bestKey = null;
    let beforeAfter = beforeMatches.intersection(afterMatches);
    let allThree = beforeAfter.intersection(ansMatches);
    let beforeAns = beforeMatches.intersection(ansMatches);
    let afterAns = afterMatches.intersection(ansMatches);
    let searchPriority = [
      allThree,
      beforeAfter,
      beforeAns,
      afterAns,
      beforeMatches,
      afterMatches,
      ansMatches
    ];
    for (let candidateSet of searchPriority) {
        if (candidateSet.size == 1) {
            bestKey = [...candidateSet][0];
            break;
        } else if (candidateSet.size > 1) {
            let best = null;
            for (let candidate of candidateSet) {
                let cID = parseInt(candidate.split('#')[1].split(':')[0])
                if (isNaN(cID)) {
                   console.warn("Invalid candidate ID: " + candidate);
                }
                // Find the one with the closest ID number
                let dist = Math.abs(cID - qID);
                if (best == null || dist < best) {
                    best = dist;
                    bestKey = candidate;
                }
            }
            break;
        } // else no matches in this candidate set; continue to the next
    }
    // If we didn't find any candidates, return null
    if (bestKey == null) {
        return null;
    }

    // Otherwise try to retrieve the answer value for our best candidate,
    // and return empty string while logging a warning if this fails.
    result = storage.getItem(bestKey);
    if (result == null) {
        console.warn("Invalid byType entry: " + bestCandidate);
        return null;
    } else {
        return bestKey;
    }
}


// Restores the displayed value of an answer from the saved value in
// local storage (see saveAnswer).
// TODO: This doesn't always seem to be working for groups... ?
function restoreSavedAnswer(qElem) {
    let qType = getQuestionType(qElem);
    let saved = retrieveAnswer(qElem);
    if (DEBUG_STORAGE) {
        let qID = getQID(qElem);
        key = QID__AKEY.get(qID);
        console.log("Restoring Saved Answer for key: ", key);
        console.log(qElem);
        console.log("Saved value is: '" + saved + "'");
    }
    if (qType == 'blank') {
        qElem.querySelector('input').value = saved;
    } else if (qType == 'option') {
        let flavor = qElem.getAttribute('data-select-flavor');
        if (flavor == "dropdown") {
            let selector = qElem.querySelector('select')
            let selected = new Set([...saved.split(';;')]);
            for (let opt of selector.options) {
                if (selected.has(opt.value)) {
                    opt.selected = true;
                } else {
                    opt.selected = false;
                }
            }
            selector.value = saved;
        } else if (flavor == "cycle") {
            // We assume only one selected answer here
            hit = false;
            for (let opt of qElem.querySelectorAll('.select-option')) {
                if (opt.innerText == saved) {
                    opt.style.display = 'inline';
                    hit = true;
                } else {
                    opt.style.display = 'none';
                }
            }
            if (!hit) {
                // Make first listed option visible by default
                if (saved != "") {
                    console.warn(
                        "Saved answer doesn't match any cycle option: " + saved
                    );
                    console.warn(qElem);
                }
                for (let opt of qElem.querySelectorAll('.select-option')) {
                    opt.style.display = 'inline';
                    break;
                }
            }
        } else {  // Same procedure for 'multiple' or 'radio'
            selected = new Set();
            for (let part of saved.split(';;')) {
                selected.add(part);
            }
            for (let opt of qElem.querySelectorAll('.select-option')) {
                let checkbox = opt.querySelector('input');
                if (checkbox === null) {
                    console.warn("Missing input element for option question:");
                    console.warn(qElem);
                } else {
                    if (selected.has(opt.innerText)) {
                        checkbox.checked = true;
                    } else {
                        checkbox.checked = false;
                    }
                }
            }
            // Selecting no answer is possible and valid
        }
    } else if (qType == 'short-answer') {
        let inp = qElem.querySelector('textarea');
        if (inp == null) {
            inp = qElem.querySelector('input');
        }
        inp.value = saved;
    } else if (
        qType == 'no-answer'
     || qType == "GROUP_START"
     || qType == "GROUP_END"
    ) {
        // Nothing to do here; we won't warn since it's reasonably to
        // call restoreSavedAnswer on "all questions"
    } else {
        console.warn(
            "Can't restore answer for question type: '" + qType + "'."
        );
    }
}


function showAnswer() {
    let qElem = this.parentElement;
    this.innerText = HIDE_ANSWER_TEXT;
    this.setAttribute('title', 'Hide Answer');
    this.setAttribute('aria-label', 'Hide Answer');
    qElem.classList.add('show');

    // Save the current answer value
    saveAnswer(qElem);

    // Update feedback visibility
    let [answer, correct] = extractAnswer(qElem);
    showFeedback(qElem, answer, true); // as if correct

    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                let sub = document.getElementById(subID);
                if (sub.classList.contains('no-check')) {
                    showFeedback(sub);
                } else {
                    setToAnswer(sub);
                }
            }
        }
    }
}

function hideAnswer() {
    let qElem = this.parentElement;
    this.innerText = SHOW_ANSWER_TEXT;
    this.setAttribute('title', "Show Answer");
    this.setAttribute('aria-label', "Show Answer");
    qElem.classList.remove('show');

    if (qElem.classList.contains('no-check')) {
        hideFeedback(qElem);
    }

    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                let sub = document.getElementById(subID);
                if (sub.classList.contains('no-check')) {
                    hideFeedback(sub);
                } else {
                    if (DEBUG_STORAGE) {
                        console.log("Restoring sub-answer for:");
                        console.log(sub);
                    }
                    restoreSavedAnswer(sub);
                }
            }
        }
    }
}

function toggleAnswer() {
    let qElem = this.parentElement;
    if (qElem.classList.contains('show')) {
        hideAnswer.call(this);
    } else {
        showAnswer.call(this);
    }
}

function showAllAnswers() {
    for (let qElem of document.querySelectorAll('.question')) {
        let button = qElem.querySelector(".show-button");
        if (button != null) {
            showAnswer.call(button);
        }
    }
}

function hideAllAnswers() {
    for (let qElem of document.querySelectorAll('.question')) {
        let button = qElem.querySelector(".show-button");
        if (button != null) {
            hideAnswer.call(button);
        }
    }
}

function restoreAllAnswers() {
    for (let qElem of document.querySelectorAll('.question')) {
        restoreSavedAnswer(qElem);
    }
}


function cycleAdvance() {
    let setNext = false;
    for (
        let opt of
        this.querySelectorAll('.select-option')
    ) {
        if (setNext) {
            opt.setAttribute('aria-selected', 'true');
            this.setAttribute(
                'aria-activedescendant',
                opt.getAttribute('id')
            );
            opt.style.display = 'inline';
            setNext = false;
        } else if (opt.getAttribute('aria-selected') == 'true') {
            setNext = true;
            opt.setAttribute('aria-selected', 'false');
            opt.style.display = 'none';
        }
	}
    if (setNext) {
        let opt = this.firstElementChild;
        opt.setAttribute('aria-selected', 'true');
        this.setAttribute(
            'aria-activedescendant',
            opt.getAttribute('id')
        );
        opt.style.display = 'inline';
	}
}


function showHints() {
    let qElem = this.parentElement;
    this.innerText = HIDE_HINT_TEXT;
    this.setAttribute('title', 'Hide Hints');
    this.setAttribute('aria-label', 'Hide Hints');
    qElem.classList.add('show-hints');
    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                let sub = document.getElementById(subID);
                showHints.call(sub.firstElementChild);
            }
        }
    }
}

function hideHints() {
    let qElem = this.parentElement;
    this.innerText = SHOW_HINT_TEXT;
    this.setAttribute('title', "Show Hints");
    this.setAttribute('aria-label', "Show Hints");
    qElem.classList.remove('show-hints');
    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                let sub = document.getElementById(subID);
                hideHints.call(sub.firstElementChild);
            }
        }
    }
}

function toggleHints() {
    let qElem = this.parentElement;
    if (qElem.classList.contains('show-hints')) {
        hideHints.call(this);
    } else {
        showHints.call(this);
    }
}


// Looks at ancestor HTML elements (including the given element) to find
// the first one that has class 'question'. Returns `null` if there is no
// such element.
function getQuestionAncestor(elem) {
    if (elem.classList.contains("question")) {
        return elem;
    } else {
        let parent = elem.parentElement;
        if (parent == null) {
            return null;
        } else {
            return getQuestionAncestor(parent);
        }
    }
}


// Delay for auto-save in milliseconds.
const AUTOSAVE_TIMEOUT_MS = 1500;

// Sets a timeout specific to the question element containing 'this' that
// will trigger a call to `saveAnswer` after `AUTOSAVE_TIMEOUT_MS`
// milliseconds. However, if that timeout already exists, it just
// refreshes it. The cumulative effect is to auto-save a bit after
// input activity stops (for at least `AUTOSAVE_TIMEOUT_MS`
// milliseconds). This prevents us from constantly auto-saving on every
// single input, but still allows us to be pretty sure that things get
// saved whenever input happens.
function autoSave(event) {
    let qElem = getQuestionAncestor(event.currentTarget);
    let timeoutID = qElem.autoSaveTimeout;
    if (timeoutID != null) {
        // clear the old timeout
        window.clearTimeout(timeoutID);
    }
    // set a new timeout
    qElem.autoSaveTimeout = window.setTimeout(
        () => { saveAnswer(qElem) },
        AUTOSAVE_TIMEOUT_MS,
        qElem
    );
}


// Adds check-answer, show-hints, and show-answer buttons to a question.
// Also sets up auto-save trigger on the question's input element(s).
function addButtons(qElem) {
    let qType = getQuestionType(qElem);
    let insertBefore = qElem.querySelector('.answer');

    if (qElem.querySelector('.hint') != null) {
        let hintButton = document.createElement('button');
        hintButton.classList.add('hint-button');
        hintButton.innerText = SHOW_HINT_TEXT;
        hintButton.setAttribute('title', "Show Hints");
        hintButton.setAttribute('aria-label', "Show Hints");
        hintButton.setAttribute('type', 'button');
        hintButton.addEventListener('click', toggleHints);
        hintButton.addEventListener('keyup', toggleHints);
        insertBefore.insertAdjacentElement('beforebegin', hintButton);
    }

    if (qType == 'short-answer' || qType == 'no-answer') {
        // no checking for short-answer and no-answer questions
        qElem.classList.add('no-check');
    } else {
        let checkButton = document.createElement('button');
        checkButton.classList.add('check-button');
        checkButton.innerText = CHECK_ANSWER_TEXT;
        checkButton.setAttribute('title', 'Check Answer');
        checkButton.setAttribute('aria-label', 'Check Answer');
        checkButton.setAttribute('type', 'button');
        checkButton.addEventListener('click', checkAnswer);
        checkButton.addEventListener('keyup', checkAnswer);
        insertBefore.insertAdjacentElement('beforebegin', checkButton);

        // Also add congrats/condole elements
        let condole = document.createElement('span');
        condole.innerText = 'Incorrect.';
        condole.classList.add('condole');
        let congrats = document.createElement('span');
        congrats.innerText = 'Correct!';
        congrats.classList.add('congrats');
        insertBefore.insertAdjacentElement('afterend', condole);
        insertBefore.insertAdjacentElement('afterend', congrats);
    }

    let showButton = document.createElement('button');
    showButton.classList.add('show-button');
    showButton.innerText = SHOW_ANSWER_TEXT;
    showButton.setAttribute('title', "Show Answer");
    showButton.setAttribute('aria-label', "Show Answer");
    showButton.setAttribute('type', 'button');
    showButton.addEventListener('click', toggleAnswer);
    showButton.addEventListener('keyup', toggleAnswer);
    insertBefore.insertAdjacentElement('beforebegin', showButton);

    let addAutoSaveTo = [ qElem ];
    let subs = qElem.getAttribute('data-sub-ids');
    if (subs) {
        for (let subID of subs.split(' ')) {
            if (subID) {
                addAutoSaveTo.push(document.getElementById(subID));
            }
        }
    }

    for (let elem of addAutoSaveTo) {
        for (
            let input
         of elem.querySelectorAll('input, select, textarea, .cycler')
        ) {
            if (input.tagName == 'INPUT') {
                input.addEventListener('click', autoSave);
                input.addEventListener('keyup', autoSave);
                input.addEventListener('change', autoSave);
                input.addEventListener('blur', autoSave);
            } else if (input.tagName == 'TEXTAREA') {
                input.addEventListener('click', autoSave);
                input.addEventListener('keyup', autoSave);
                input.addEventListener('change', autoSave);
                input.addEventListener('blur', autoSave);
            } else if (input.tagName == 'SELECT') {
                input.addEventListener('change', autoSave);
                input.addEventListener('blur', autoSave);
            } else {
                // cycler
                input.addEventListener('click', autoSave);
                input.addEventListener('keyup', autoSave);
                input.addEventListener('blur', autoSave);
            }
        }
    }
}


// Returns a list of two strings containing a certain number of
// characters of text context from before and after the specified
// element.
function textContext(qElem, size) {
    let prevText = "";
    let prev = qElem.previousSibling;
    while (prevText.length < size && prev != null) {
        prevText = prev.innerText + prevText;
        prev = prev.previousSibling;
    }
    prevText = prevText.slice(-size);
 
    let afterText = "";
    let after = qElem.nextSibling;
    while (afterText.length < size && after != null) {
        afterText += after.innerText;
        after = after.nextSibling;
    }
    afterText = afterText.slice(0, size);
    return [ prevText, afterText ];
}


// Hashing function for strings from:
// https://github.com/bryc/code/blob/master/jshash/experimental/cyrb53.js

function cyrb53(str, seed = 0) {
    let h1 = 0xdeadbeef ^ seed, h2 = 0x41c6ce57 ^ seed;
    for(let i = 0, ch; i < str.length; i++) {
        ch = str.charCodeAt(i);
        h1 = Math.imul(h1 ^ ch, 2654435761);
        h2 = Math.imul(h2 ^ ch, 1597334677);
    }
    h1  = Math.imul(h1 ^ (h1 >>> 16), 2246822507);
    h1 ^= Math.imul(h2 ^ (h2 >>> 13), 3266489909);
    h2  = Math.imul(h2 ^ (h2 >>> 16), 2246822507);
    h2 ^= Math.imul(h1 ^ (h1 >>> 13), 3266489909);
  
    return 4294967296 * (2097151 & h2) + (h1 >>> 0);
}


// Global variable for assigning question identities.
var nextID = 0;

// Sets custom attributes on a question to assign it an identity
function assignIdentity(qElem) {
    var thisID = nextID;
    nextID += 1;
    let [thisBefore, thisAfter] = textContext(qElem, 15);
    let thisAnswer = getCorrectAnswer(qElem);
    qElem.setAttribute("data-qID", thisID);
    qElem.setAttribute("data-ctxBefore", cyrb53(thisBefore));
    qElem.setAttribute("data-ctxAfter", cyrb53(thisAfter));
    qElem.setAttribute("data-ctxAnswer", cyrb53(thisAnswer));
}


// Returns an array with 5 elements: a running numerical unique ID
// number, the question type string, and hashes for the answer and for 15
// characters of context before and after the target element. These are
// read from the data- attributes of the element as set by `assignIdentity`.
function getIdentity(qElem) {
    return [
        qElem.getAttribute("data-qID"),
        qElem.getAttribute('data-question-type'),
        qElem.getAttribute("data-ctxAnswer"),
        qElem.getAttribute("data-ctxBefore"),
        qElem.getAttribute("data-ctxAfter")
    ];
}


// Gets the question key for a question element, used for storing &
// retrieving answers.
function getQuestionKey(qElem) {
    let [qID, qType, ansID, ctxBefore, ctxAfter] = getIdentity(qElem);
    let urlPath = new URL(document.URL).pathname;
    return (
        urlPath
      + "#" + qID
      + ":" + qType
      + ":" + ansID
      + ":" + ctxBefore
      + ":" + ctxAfter
    );
}

// Gets the question ID for a question element. These are assigned during
// setup.
function getQID(qElem) {
    return qElem.getAttribute("data-qID");
}

// Gets the question type for a question element.
function getQuestionType(qElem) {
    return qElem.getAttribute('data-question-type');
}


// Mappings from question IDs to answer keys and back, which we fill in
// during the setup process. Necessary because we want to be robust
// against adding/removing/changing questions when we store answers.
// We update this whenever we save an answer if necessary.
QID__AKEY = new Map();
AKEY__QID = new Map();


// Matches all question IDs with answer keys stored in local storage.
// Uses findAnswerKey to find matches one-by-one in cases where exact
// matches aren't found.
function matchAnswerKeys() {
    // First, check for exact matches
    let allQs = document.querySelectorAll('.question');
    for (let qElem of allQs) {
        let qID = getQID(qElem);
        let perfect = findAnswerKey(qElem, AKEY__QID, true);
        if (perfect != null) {
            AKEY__QID.set(perfect, qID);
            QID__AKEY.set(qID, perfect);
        }
    }
    // Next check for inexact matches for unmatched questions
    for (let qElem of allQs) {
        let qID = getQID(qElem);
        if (!QID__AKEY.has(qID)) {
            let best = findAnswerKey(qElem, AKEY__QID, false);
            if (best != null) {
                AKEY__QID.set(best, qID);
                QID__AKEY.set(qID, best);
            }
        }
    }
}


function setup() {
    // 1. Group multiple-choice options together.
    // 2. Graft hints, feedback, and explanations onto their parent
    //    questions.
    // 3. Find start/end group 'questions' and aggregate buttons.
    // 4. Call addButtons and assignIdentity for each grouped question,
    //    plus assignIdentity for each subquestion.
    // 5. Create a 'show all answers' button and add it to the end of the
    //    page (or as a child of the '#discovery-show-all' element if
    //    there is one).
    // 6. Set up mappings between question IDs and answer keys.
    // 7. Restore all saved answers.

    // Group options together & process unit rules
    for (let elem of document.querySelectorAll('.question')) {
        let headID = elem.getAttribute('id');
        if (getQuestionType(elem) == 'UNITS') {
            let target = elem.parentElement;
            elem.remove();
            while (target.childNodes.length == 1) {
                let del = target;
                target = target.parentElement;
                del.remove();
            }
            // Add unit rule
            let regex = buildQuestionRegex(elem);
            let repl = elem.querySelector('.answer').innerText;
            UNIT_CONVERSIONS.push([regex, repl]);
        } else if (
            elem.parentElement
         && getQuestionType(elem) == 'option'
        ) {
            let flavor = elem.getAttribute('data-select-flavor');
            let ans = document.createElement('div');
            ans.classList.add('answer');
            ans.innerHTML = "Correct Answer(s): <br>"
            let firstOption = elem.firstElementChild;
            if (firstOption) {
                firstOption.setAttribute('name', headID);
                if (firstOption.classList.contains('correct')) {
                    let holder = document.createElement('div');
                    holder.innerHTML = firstOption.outerHTML;  // clone it
                    // remove checkbox
                    holder.firstElementChild.firstElementChild.remove();
                    holder.firstElementChild.classList.remove('select-option');
                    ans.append(holder.firstElementChild);  // transfer clone
                }
            }

            next = elem.nextElementSibling;

            while (
                next
             && getQuestionType(next) == 'option'
            ) {
                let thisID = next.getAttribute('id');
                if (!flavor) {
                    flavor = next.getAttribute('data-select-flavor');
                }
                let option = next.firstElementChild;
                option.remove();
                elem.append(option);
                if (thisID) {
                    option.setAttribute('id', thisID);
                }
                option.setAttribute('name', headID);
                if (option.classList.contains('correct')) {
                    let holder = document.createElement('div');
                    holder.innerHTML = option.outerHTML;  // clone it
                    // remove checkbox
                    holder.firstElementChild.firstElementChild.remove();
                    holder.firstElementChild.classList.remove(
                        'select-option'
                    );
                    ans.append(holder.firstElementChild);  // transfer clone
                }

                dispose = next;
                next = next.nextElementSibling;
                dispose.remove();
            }

            // Re-format by flavor:
            if (flavor == 'radio') {
                for (let option of elem.querySelectorAll('.select-option')) {
                    let replacing = option.firstElementChild;
                    let repID = option.getAttribute('id');
                    let repName = option.getAttribute('name');
                    let rb = document.createElement('input');
                    rb.setAttribute('type', 'radio');
                    if (repID) {
                        rb.setAttribute('id', repID);
                    }
                    rb.setAttribute('name', repName);
                    option.replaceChild(rb, replacing);
                }
            } else if (flavor == 'dropdown') {
                let sel = document.createElement('select');
                for (let option of elem.querySelectorAll('.select-option')) {
                    option.remove();
                    let opt = document.createElement('option');
                    opt.setAttribute('value', option.innerText);
                    opt.innerText = option.innerText;
                    if (option.classList.contains('correct')) {
                        opt.classList.add('correct');
                    }
                    sel.append(opt);
                }
                elem.append(sel);
            } else if (flavor == 'cycle') {
                let cycler = document.createElement('a');
                cycler.classList.add('cycler');
                for (let option of elem.querySelectorAll('.select-option')) {
                    option.remove();
                    let opt = document.createElement('span');
                    opt.classList.add('select-option');
                    opt.setAttribute('data-value', option.innerText);
                    opt.setAttribute('role', 'option');
                    opt.setAttribute('aria-selected', false);
                    opt.style.display = 'none';
                    opt.innerText = option.innerText;
                    if (option.classList.contains('correct')) {
                        opt.classList.add('correct');
                    }
                    cycler.append(opt);
                }
                let selected = cycler.firstElementChild;
                selected.style.display = "inline";
                selected.setAttribute('aria-selected', true);
                cycler.setAttribute('role', 'listbox');
                cycler.setAttribute('tabindex', '0');
                cycler.setAttribute(
                    'aria-label',
                    "(click to select the next option)"
                );
                cycler.addEventListener('click', cycleAdvance);
                cycler.addEventListener('keyup', cycleAdvance);
                elem.append(cycler);
            } // else no change needed flavor is 'multiple' or undefined

            // Set data select flavor
            elem.setAttribute('data-select-flavor', flavor);

            // Add answer at end
            elem.append(ans);
        }
    }

    // Graft hints/feedback/explanations
    let allQs = document.querySelectorAll('.question');
    for (let elem of allQs) {
        if (!elem.parentElement) { continue; }  // already removed from DOM
        let insertAfter = elem.querySelector('.answer');
        next = elem.nextElementSibling;
        while (
            next
         && next.classList.contains('question')
         && (
                getQuestionType(next) == 'hint'
             || getQuestionType(next) == 'feedback'
             || getQuestionType(next) == 'explanation'
            )
        ) {
            let thisID = next.getAttribute('id');
            let iType = getQuestionType(next);
            let content = next.querySelector('.answer');
            content.classList.add(iType);
            content.classList.remove('answer');
            let flags = next.getAttribute('data-flags');
            content.setAttribute('data-flags', flags);
            let regex = next.getAttribute('data-regex');
            if (regex != null) {
                content.setAttribute('data-regex', regex);
                content.setAttribute(
                    'data-regex-flags',
                    next.getAttribute('data-regex-flags')
                );
            }

            insertAfter.insertAdjacentElement('afterend', content);
            insertAfter = content;

            let dispose = next;
            next = next.nextElementSibling;
            dispose.remove();
        }
    }

    // Group questions in between _G / _E grouping items.
    allQs = document.querySelectorAll('.question');
    let inGroup = null;
    let grouped = [];
    for (let elem of allQs) {
        let qType = getQuestionType(elem);
        if (inGroup === null) {
            if (qType == 'GROUP_START') {
                inGroup = elem.getAttribute('id');
            }  // else do nothing
        } else {
            if (qType == 'GROUP_END') {
                let subsStr = '';
                for (let sub of grouped) {
                    let subID = sub.getAttribute('id');
                    subsStr += ' ' + subID;
                }
                elem.setAttribute('data-sub-ids', subsStr);
                elem.setAttribute('data-group-id', inGroup);
                inGroup = null;
                grouped = [];
            } else {
                grouped.push(elem);
                elem.setAttribute('data-within-group', inGroup);
                elem.classList.add('subquestion');
            }
        }
    }

    // Re-select after grouping to add buttons
    allQs = document.querySelectorAll('.question');
    for (let elem of allQs) {
        let qType = getQuestionType(elem);
        // Change p -> span -> input[type=text] into div -> textarea
        if (
            getQuestionType(elem) == 'short-answer'
         && elem.parentElement.childNodes.length == 1
         && elem.parentElement.tagName == 'P'
        ) {
            let replacement = document.createElement('div');
            replacement.classList.add('question');
            for (let attr of ['id', 'data-question-type', 'data-flags']) {
                replacement.setAttribute(attr, elem.getAttribute(attr));
            }
            if (elem.hasAttribute('data-regex')) {
                replacement.setAttribute(
                    'data-regex',
                    elem.getAttribute('data-regex')
                );
                replacement.setAttribute(
                    'data-regex-flags',
                    elem.getAttribute('data-regex-flags')
                );
            }
            replacement.innerHTML = elem.innerHTML;  // clone innards
            let newInput = document.createElement('textarea');
            replacement.replaceChild(
                newInput,
                replacement.querySelector('input')
            );
            elem.parentElement.parentElement.replaceChild(
                replacement,
                elem.parentElement
            );
            if (!elem.classList.contains('subquestion')) {
                addButtons(replacement);
            }
            assignIdentity(replacement);
        } else {
            if (qType == 'GROUP_START') {
                elem.remove();
            } else if (!elem.classList.contains('subquestion')) {
                addButtons(elem);
                assignIdentity(elem);
            } else {
                assignIdentity(elem);
            }
        }
    }

    // Create our show-all and hide-all buttons
    var showAllButton = document.createElement('button');
    showAllButton.innerText = "Show all answers"
    showAllButton.addEventListener(
        "click",
        showAllAnswers
    );
    var hideAllButton = document.createElement('button');
    hideAllButton.innerText = "Hide all answers"
    hideAllButton.addEventListener(
        "click",
        hideAllAnswers
    );
    var addWhere = document.getElementById("discovery-show-all");
    if (addWhere == null) {
        addWhere = document.body;
    }
    addWhere.appendChild(showAllButton);
    addWhere.appendChild(document.createTextNode(' '));
    addWhere.appendChild(hideAllButton);

    // Fill in our QID <-> AKEY mappings
    matchAnswerKeys();

    // Restore all saved answers
    restoreAllAnswers();
}


// Runs all of our unit tests.
function test() {
    testDeprefix();
    testStandardizeUnits();
    testCompareAnswer();
}


window.onload = setup;
'''


ID_N = 0  # next ID number to assign


def nextID():
    """
    Returns a new unique question ID.
    """
    global ID_N
    result = 'q' + str(ID_N)
    ID_N += 1
    return result


def grabRE(text, start):
    """
    Given the index within some text of a starting '/' for a regular
    expression attached to a question, returns a tuple containing the
    regular expression string within the slashes, any option letters for
    that expression, and the index of the last character of the regular
    expression (might be after a '/'). Returns `None` if there is no
    matching closing square bracket (or if the specified index is not an
    open bracket).

    Examples:

    >>> grabRE('/abc/', 0)
    ('abc', '', 4)
    >>> grabRE('/abc/i', 0)
    ('abc', 'i', 5)
    >>> grabRE('/abc', 0) is None
    True
    >>> grabRE('/abc/', 1) is None
    True
    >>> grabRE('/abc def/', 0)
    ('abc def', '', 8)
    """
    if text[start] != '/':
        return None

    escaped = False
    inOptions = False
    expr = ''
    options = ''
    for i in range(start + 1, len(text)):
        c = text[i]
        if inOptions:
            if c.isalpha():
                options += c
            else:
                return (expr, options, i - 1)
        elif escaped:
            escaped = False
            expr += c
        elif c == '\\':
            escaped = True
        elif c == '/':
            inOptions = True
        else:
            expr += c

    if inOptions:
        return (expr, options, i)
    else:
        return None


# Globals for parsing stuff
foundQuestions: List[Dict] = []
currentQuestion: Dict[str, Any] = {}


def hasAncestor(elem, ancestor):
    """
    Returns True if the given element has the other given element as an
    ancestor, and False otherwise. Always returns False if one or both
    arguments is `None`.
    """
    if elem is None or ancestor is None:
        return False

    if elem.parent == ancestor:
        return True
    else:
        return hasAncestor(elem.parent, ancestor)


def allAncestors(elem):
    """
    Yields the list of all ancestors of the given element, starting
    with itself. Yields nothing if given `None`.
    """
    if elem is None:
        return
    else:
        yield elem
        yield from allAncestors(elem.parent)


def commonAncestor(startElem, endElem):
    """
    Returns the earliest common ancestor of the two elements, or `None`
    if they have no common ancestors. If one element is an ancestor of
    the other, that element will be returned. If the two elements are
    the same, that element will be returned.

    For example:


    """
    startAncestors = set(id(x) for x in allAncestors(startElem))
    for elem in allAncestors(endElem):
        thisID = id(elem)
        if thisID in startAncestors:
            return elem

    return None


def divable(elem):
    """
    Given an element, returns it, or if it was not a Block element,
    returns it wrapped in a Plain.
    """
    if not isinstance(elem, pf.Block):
        elem = pf.Plain(elem)
    return elem


def classable(elem):
    """
    Given an element, returns it, or if it was not a classable element,
    returns it wrapped in a Span.
    """
    if not hasattr(elem, 'classes'):
        elem = pf.Span(elem)
    return elem


def createQuestionBetweenStrings(
    questionInfo,
    startIndex,
    answerStart,
    startElem,
    endElem,
    answerEnd,
    endIndex
):
    """
    Alters the element tree, replacing parts between two Str elements
    with a new Span element representing a question w/ included answer.
    Note that Javascript will come by later and replace this structure
    with the proper interactive one; if Javascript does not activate,
    the result will be HTML code that shows an input element for the
    question and which has a CSS-hidden answer element grouped with it
    in a span (or possibly just a hidden hint/explanation/feedback
    span).

    Returns the newly-created element (which also gets inserted into the
    existing document structure).

    For example:

    >>> import panflute as pf
    >>> qi = {
    ...     'spec': '___',
    ...     'answer': pf.Str('answer'),
    ... }
    >>> a = pf.Str('hi[___][|AA')
    >>> z = pf.Str('ZZ|]bye')
    >>> bet = pf.Span(pf.Str('in'), pf.Space(), pf.Str('between'))
    >>> par = pf.Para(
    ...    pf.Span(pf.Str('before'), a, pf.Str('in'), classes=['K']),
    ...    bet,
    ...    z
    ... )
    >>> a.parent = par.content[0]
    >>> z.parent = par
    >>> bet.parent = par
    >>> created = createQuestionBetweenStrings(
    ...     qi,
    ...     2,
    ...     'AA',
    ...     a,
    ...     z,
    ...     'ZZ',
    ...     3,
    ... )
    >>> created == pf.Span(
    ...     pf.RawInline('<input type="text">', format='html'),
    ...     pf.RawInline('</input>', format='html'),
    ...     pf.Span(
    ...         pf.Span(
    ...             pf.Span(
    ...                 pf.Str('AA'),
    ...                 pf.Str('in'),
    ...                 classes=['K']
    ...             ),
    ...             bet,
    ...             pf.Str('ZZ')
    ...         ),
    ...         classes=['answer']
    ...     ),
    ...     classes=['question'],
    ...     attributes={'data-flags': ''}
    ... )
    True
    >>> par == pf.Para(
    ...    pf.Span(pf.Str('before'), pf.Str('hi'), classes=['K']),
    ...    created,
    ...    pf.Str('bye')
    ... )
    True
    """
    # Get common ancestor and chains of descent from it:
    ancestor = commonAncestor(startElem, endElem)
    # This fixes up some otherwise broken properties for our unit tests...
    ancestor.walk(lambda *x: None)
    startAncestry = []
    for parent in allAncestors(startElem):
        if parent is ancestor:
            break
        else:
            startAncestry.append(parent)

    startAncestry = startAncestry[::-1]

    endAncestry = []
    for parent in allAncestors(endElem):
        if parent is ancestor:
            break
        else:
            endAncestry.append(parent)
    endAncestry = endAncestry[::-1]

    # Create new Span element for the answer part of our question
    answerElems = []

    # First, pull out parts of the start ancestry that are after the
    # start element, and parts of the end ancestry that are before the
    # end element.
    foreInto = answerElems
    foreParent = None
    aftInto = answerElems
    aftParent = None
    gotBetween = False
    for depth in range(max(len(startAncestry), len(endAncestry))):
        foreTarget = None
        aftTarget = None
        if depth < len(startAncestry):
            foreTarget = startAncestry[depth]
        if depth < len(endAncestry):
            aftTarget = endAncestry[depth]

        if foreTarget is not None:
            foreClone = copy.deepcopy(foreTarget)
            if hasattr(foreClone, 'content'):
                foreClone.content = []
            foreInto.append(foreClone)
            foreClone.parent = foreParent

            if foreTarget is startElem:  # modify text to split node
                foreClone.text = answerStart
                startElem.text = startElem.text[:startIndex]

            if aftTarget is not foreTarget:
                if aftTarget is None:
                    grabUntil = len(foreTarget.parent.content)
                    grabFrom = False
                else:
                    if not gotBetween:
                        # only the first divergent level needs to grab
                        # between stuff
                        gotBetween = True
                        grabUntil = aftTarget.index
                        grabFrom = False
                        # here aftInto is the same node as foreInto
                    else:
                        # grab stuff after to end of contents
                        grabUntil = len(foreTarget.parent.content)
                        grabFrom = True

                for i in range(foreTarget.index + 1, grabUntil):
                    between = foreTarget.parent.content.pop(
                        foreTarget.index + 1
                    )  # index doesn't change as things get popped
                    foreInto.append(between)
                    between.parent = foreParent

                if grabFrom:
                    for i in range(aftTarget.index):
                        between = aftTarget.parent.content.pop(0)
                        aftInto.append(between)
                        between.parent = aftParent

                # Only if they're different do we need to clone the aft one
                if aftTarget is not None:
                    aftClone = copy.deepcopy(aftTarget)
                    if hasattr(aftClone, 'content'):
                        aftClone.content = []
                    aftInto.append(aftClone)
                    aftClone.parent = aftParent
                    if aftTarget is endElem:  # modify text to split node
                        aftClone.text = answerEnd
                        endElem.text = endElem.text[endIndex + 1:]

            # If the two targets are the same, there's nothing in-between at
            # this level

        elif aftTarget is not None:  # no fore target but an aft one
            if gotBetween:
                for i in range(aftTarget.index):
                    between = aftTarget.parent.content.pop(0)
                    aftInto.append(between)
                    between.parent = aftParent

            aftClone = copy.deepcopy(aftTarget)
            if hasattr(aftClone, 'content'):
                aftClone.content = []
            aftInto.append(aftClone)
            aftClone.parent = aftParent
            if aftTarget is endElem:  # modify text to split node
                aftClone.text = answerEnd
                endElem.text = endElem.text[endIndex + 1:]

        # Cycle down to next level
        if foreTarget is not None:
            try:
                foreInto = foreClone.content
                foreParent = foreClone
            except AttributeError:  # if we reached a Str or similar
                foreInto = None
                foreParent = None
        else:
            foreInto = None
            foreParent = None

        if aftTarget is not None:
            try:
                aftInto = aftClone.content
                aftParent = aftClone
            except AttributeError:  # if we reached a Str or similar
                aftInto = None
                aftParent = None
        else:
            aftInto = None
            aftParent = None

    # Now we've ripped out and/or cloned stuff in between as part of the
    # new answer, so we can create our question element

    # Figure out if we need to wrap our answer as a Div or Span
    if any(isinstance(x, pf.Block) for x in answerElems):
        answer = pf.Div(*map(divable, answerElems))
    else:
        answer = pf.Span(*answerElems)

    for elem in answerElems:
        elem.parent = answer

    if hasattr(ancestor, 'classes'):
        answer.classes = copy.deepcopy(ancestor.classes)
    if hasattr(ancestor, 'attributes'):
        answer.attributes = copy.deepcopy(ancestor.attributes)

    questionInfo['answer'] = answer

    # Call out to createQuestionElement to build the question
    newElem = createQuestionElement(questionInfo)

    # Ensure the question is an appropriate type for fitting into the
    # common ancestor
    # TODO: Better way of testing whether or not the new element needs to
    # be wrapped in a Plain or not!
    if isinstance(ancestor, pf.Div):
        newElem = divable(newElem)

    # Finally, place it back into the common ancestor, between the start
    # and end.
    ancestor.content.insert(startAncestry[0].index + 1, newElem)

    return newElem


def test_cqbs():
    """
    Extra tests for `createQuestionBetweenStrings`.
    """
    # Fake question info dict:
    qi = {
        'spec': '___',
        'answer': pf.Str('answer'),
    }

    # Start & end elements
    a = pf.Str('hi[___][|AA')
    z = pf.Str('ZZ|]bye')
    # Paragraph with just those two
    par = pf.Para(a, z)
    # Fix parents
    a.parent = par
    z.parent = par

    # Create question
    created = createQuestionBetweenStrings(
        qi,
        2,
        'AA',
        a,
        z,
        'ZZ',
        3,
    )

    # Checks
    assert created == pf.Span(
        pf.RawInline('<input type="text">', format='html'),
        pf.RawInline('</input>', format='html'),
        pf.Span(
            pf.Span(
                pf.Str('AA'),
                pf.Str('ZZ'),
            ),
            classes=['answer']
        ),
        classes=['question'],
        attributes={'data-flags': ''}
    )
    assert par == pf.Para(
        pf.Str('hi'),
        created,
        pf.Str('bye')
    )

    # New (identical) start/end elements
    a = pf.Str('hi[___][|AA')
    z = pf.Str('ZZ|]bye')
    # Something in between
    bet = pf.Span(pf.Str('in'), pf.Space(), pf.Str('between'))
    # Div w/ those elements
    div = pf.Div(pf.Plain(a, bet, z))
    # Fix parents
    a.parent = div.content[0]
    z.parent = div.content[0]
    bet.parent = div.content[0]

    # Tests
    created2 = createQuestionBetweenStrings(
        qi,
        2,
        'AA',
        a,
        z,
        'ZZ',
        3,
    )
    assert created2 == pf.Span(
        pf.RawInline('<input type="text">', format='html'),
        pf.RawInline('</input>', format='html'),
        pf.Span(
            pf.Span(
                pf.Str('AA'),
                bet,
                pf.Str('ZZ'),
            ),
            classes=['answer']
        ),
        classes=['question'],
        attributes={'data-flags': ''}
    )
    assert div == pf.Div(pf.Plain(pf.Str('hi'), created2, pf.Str('bye')))

    # Re-do setup
    a = pf.Str('hi[___][|AA')
    z = pf.Str('ZZ|]bye')
    bet = pf.Span(pf.Str('in'), pf.Space(), pf.Str('between'))

    # This time end of question is deeper
    par3 = pf.Para(
       a,
       bet,
       pf.Span(pf.Str('in'), z, pf.Str('after'), classes=['K']),
    )
    a.parent = par3
    z.parent = par3.content[2]
    bet.parent = par3

    # Tests
    created3 = createQuestionBetweenStrings(
        qi,
        2,
        'AA',
        a,
        z,
        'ZZ',
        3,
    )
    assert created3 == pf.Span(
        pf.RawInline('<input type="text">', format='html'),
        pf.RawInline('</input>', format='html'),
        pf.Span(
            pf.Span(
                pf.Str('AA'),
                bet,
                pf.Span(
                    pf.Str('in'),
                    pf.Str('ZZ'),
                    classes=['K']
                ),
            ),
            classes=['answer']
        ),
        classes=['question'],
        attributes={'data-flags': ''}
    )

    assert par3 == pf.Para(
       pf.Str('hi'),
       created3,
       pf.Span(pf.Str('bye'), pf.Str('after'), classes=['K']),
    )

    # Full document to test on
    doc = pf.convert_text(
        'Top\n\n[_...][|Short\nanswer.|]\n\nBottom',
        input_format='markdown',
        output_format='panflute',
        standalone=True
    )
    created4 = createQuestionBetweenStrings(
        {'spec': '_...', 'answer': None},
        0,
        'Short',
        doc.content[1].content[0],
        doc.content[1].content[2],
        'answer.',
        8,
    )
    assert created4 == pf.Span(
        pf.RawInline('<input type="text">', format='html'),
        pf.RawInline('</input>', format='html'),
        pf.Span(
            pf.Span(
                pf.Str('Short'),
                pf.SoftBreak(),
                pf.Str('answer.'),
            ),
            classes=['answer']
        ),
        classes=['question'],
        attributes={'data-flags': ''}
    )
    assert doc.content[0] == pf.Para(pf.Str('Top'))
    assert doc.content[1] == pf.Para(pf.Str(''), created4, pf.Str(''))
    assert doc.content[2] == pf.Para(pf.Str('Bottom'))
    assert doc == pf.Doc(
        pf.Para(pf.Str('Top')),
        pf.Para(pf.Str(''), created4, pf.Str('')),
        pf.Para(pf.Str('Bottom')),
    )


def createQuestionElement(questionInfo):
    """
    Given a question info dict with 'spec' and 'answer' keys, creates a
    Panflute element to represent the question, including raw HTML for
    the correct input element and a CSS-hidden span for the answer (note
    that the output format must be HTML for these to work, and ideally,
    Javascript will package them into an interactive element).

    For 'short-answer' questions, the input element will be either a
    text input or a text area, depending on whether the answer is a
    block element or an inline element.
    """
    spec = questionInfo['spec']
    addAnswer = True

    resultType = pf.Span
    packager = lambda x: x
    attrs = {'id': nextID()}
    answerPrefix = "Correct answer: "
    if isinstance(questionInfo['answer'], pf.Block):
        resultType = pf.Div
        packager = divable

    flagsEnd = len(spec)
    try:
        flagsEnd = spec.index(']')
    except:
        pass

    if spec.startswith('_G'):
        flags = spec[2:flagsEnd]
        attrs['data-question-type'] = 'GROUP_START'
        inputElems = [pf.Span(pf.Str('G'), classes=['groupStart'])]
    elif spec.startswith('_E'):
        flags = spec[2:flagsEnd]
        attrs['data-question-type'] = 'GROUP_END'
        useText = questionInfo.get('answer')
        if not useText:
            useText = pf.Str("Check Answers")
        inputElems = [pf.Span(useText, classes=['groupEnd'])]
        questionInfo['answer'] = pf.Str("See above")
    elif spec.startswith('_U'):
        flags = spec[2:flagsEnd]
        attrs['data-question-type'] = 'UNITS'
        inputElems = []
    elif spec.startswith('___'):
        flags = ''
        attrs['data-question-type'] = 'blank'
        for i in range(len(spec)):
            if spec[i] != '_':
                flags = spec[i:flagsEnd]
                break
        inputElems = [
            pf.RawInline('<input type="text">', format='html'),
            pf.RawInline('</input>', format='html')
        ]

    elif spec.startswith('_...') or spec.startswith('_…'):
        flags = spec[4:flagsEnd]
        attrs['data-question-type'] = 'short-answer'
        answerPrefix = "Example answer: "
        # If we're in an inline context, need to use normal text input,
        # but we hack it if there's an empty paragraph as the context by
        # modifying our context post-hoc.
        if resultType == pf.Div:
            inputElems = [pf.RawBlock('<textarea></textarea>', format='html')]
        else:
            inputElems = [
                pf.RawInline('<input type="text">', format='html'),
                pf.RawInline('</input>', format='html')
            ]
    elif spec.startswith('_.'):
        flags = spec[2:flagsEnd]
        attrs['data-question-type'] = 'no-answer'
        answerPrefix = "Example answer: "
        inputElems = []
    elif (
        spec.startswith('_h')
     or spec.startswith('_f')
     or spec.startswith('_e')
    ): # hints, feedback, and explanations
        flags = spec[2:flagsEnd]
        qType = spec[1]
        if qType == 'h':
            answerPrefix = "Hint: "
            attrs['data-question-type'] = 'hint'
        elif qType == 'f':
            answerPrefix = "Feedback: "
            attrs['data-question-type'] = 'feedback'
        elif qType == 'e':
            answerPrefix = "Explanation: "
            attrs['data-question-type'] = 'explanation'
        else:
            raise RuntimeError("Impossible qType.")
        inputElems = []
    else:  # must be a multiple-choice
        flags = spec[2:flagsEnd]
        attrs['data-question-type'] = 'option'
        optType = spec[1]
        addAnswer = False  # 'answer' is actually option to display
        inputElems = [resultType(
            pf.RawInline('<input type="checkbox">', format='html'),
            pf.RawInline('</input>', format='html'),
            questionInfo['answer'],
            pf.RawInline('<br>', format='html'),
            classes=['select-option']
        )]
        if optType == '_':  # a wrong-answer; we don't know group type
            pass
        elif optType == 'x':  # a right-answer in a multiple-select
            inputElems[0].classes.append('correct')
            attrs['data-select-flavor'] = 'multiple'
        elif optType == 'r':  # a right-answer in a radio-select
            inputElems[0].classes.append('correct')
            attrs['data-select-flavor'] = 'radio'
        elif optType == 'd':  # a right-answer in a drop-down select
            inputElems[0].classes.append('correct')
            attrs['data-select-flavor'] = 'dropdown'
        elif optType == 'c':  # a right-answer in a cycle select
            inputElems[0].classes.append('correct')
            attrs['data-select-flavor'] = 'cycle'
        else:
            pf.debug(f"Unrecognized option type '{optType}'.")

    attrs['data-flags'] = flags

    # May be either a Div or a Span depending on what the answer is.
    result = resultType(
        *map(packager, inputElems),
        classes=['question'],
        attributes=attrs
    )

    if addAnswer:
        packagedAnswer = packager(questionInfo['answer'])
        if not hasattr(packagedAnswer, 'classes'):
            packagedAnswer = pf.Span(packagedAnswer)
            # TODO: Fix unit tests to account for this!
        packagedAnswer.classes.append('answer-value')
        result.content.append(
            resultType(
                packager(pf.Str(answerPrefix)),
                packagedAnswer,
                classes=['answer']
            )
        )

    if 'regex' in questionInfo:
        result.attributes['data-regex'] = questionInfo['regex']
        result.attributes['data-regex-flags'] = questionInfo['regex-flags']

    return result


def action(elem, doc):
    """
    `panflute` action function will be applied to each element in the
    document.

    Uses global variables to track state.

    Looks for question-start symbol strings (starting with '[_') among
    Str elements, and then for matching '|]' answer ends possibly in
    """
    global foundQuestions, currentQuestion

    # Look for question start/end stuff if it's a Str:
    if isinstance(elem, pf.Str):
        text = elem.text
        i = -1  # using pre-increment strategy
        while i < len(text):
            i += 1  # increment i
            endHere = None
            if currentQuestion and text[i:i + 2] == '|]':
                # Found the end of an answer we're accumulating
                answerEnd = text[:i]
                endIndex = i + 1
                newElem = createQuestionBetweenStrings(
                    currentQuestion,
                    currentQuestion['startIndex'],
                    currentQuestion['answerStart'],
                    currentQuestion['startElem'],
                    elem,
                    answerEnd,
                    endIndex
                )
                # (new element is already inserted into tree)
                # Accumulate and reset right now
                foundQuestions.append(currentQuestion)
                currentQuestion = {}
            elif text[i:i + 2] == '[_':  # possible question start
                start = i
                try:
                    specEnd = text.index(']', start)
                except ValueError:
                    raise ValueError(
                        f"Question end not found in text:\n'{text}'"
                        f"\n(found '[_' at index {i}."
                    )
                if (
                    (text[start:specEnd + 1] in ('[_G]', '[_E]'))
                and (
                        len(text[specEnd + 1:specEnd + 2]) == 0
                     or (text[specEnd + 1:specEnd + 2] not in '[/')
                    )
                ):
                    # Has no answer part
                    # Add a new question now if we haven't already:
                    if (
                        currentQuestion
                    and (
                            list(currentQuestion.keys()) != [
                                'regex',
                                'regex-flags'
                            ]
                        )
                    ):
                        foundQuestions.append(currentQuestion)
                        currentQuestion = {}
                    currentQuestion['spec'] = text[start + 1:specEnd]
                    currentQuestion['answer'] = pf.Str('')
                    # Break up string...
                    insertAt = elem.index + 1
                    before = text[:start]
                    after = text[specEnd + 1:]
                    elem.text = before  # truncate
                    newElem = createQuestionElement(currentQuestion)
                    newStr = pf.Str(after)  # rest in a new string
                    # Insert new elements
                    elem.parent.content.insert(insertAt, newStr)
                    elem.parent.content.insert(insertAt, newElem)
                    # Accumulate and reset right now
                    foundQuestions.append(currentQuestion)
                    currentQuestion = {}
                if text[specEnd + 1:specEnd + 2] not in '[/':
                    continue  # doesn't count if without '[' or '/'
                if text[specEnd + 1:specEnd + 2] == '/':
                    rexpr = grabRE(text, specEnd + 1)
                    if rexpr is None:
                        continue  # doesn't count with mis-formatted '/'
                    else:
                        # Create new question at this point
                        if currentQuestion:
                            foundQuestions.append(currentQuestion)
                            currentQuestion = {}
                        currentQuestion['regex'] = rexpr[0]
                        currentQuestion['regex-flags'] = rexpr[1]
                        specEnd = rexpr[-1]  # update end past RE
                if text[specEnd + 2:specEnd + 3] != '|':
                    if list(currentQuestion.keys()) == [
                        'regex',
                        'regex-flags'
                    ]:
                        currentQuestion = {}  # reset it
                    continue  # lack of '[|' for answer
                else:
                    # Okay add a new question now if we haven't already:
                    if (
                        currentQuestion
                    and (
                            list(currentQuestion.keys()) != [
                                'regex',
                                'regex-flags'
                            ]
                        )
                    ):
                        foundQuestions.append(currentQuestion)
                        currentQuestion = {}
                    currentQuestion['spec'] = text[start + 1:specEnd]
                    try:
                        endHere = text.index('|]', specEnd + 2)
                        currentQuestion['answer'] = pf.Str(
                            text[specEnd + 3:endHere]
                        )
                        # Break up string...
                        insertAt = elem.index + 1
                        before = text[:start]
                        after = text[endHere + 2:]
                        elem.text = before  # truncate
                        newElem = createQuestionElement(currentQuestion)
                        newStr = pf.Str(after)  # rest in a new string
                        # Insert new elements
                        elem.parent.content.insert(insertAt, newStr)
                        elem.parent.content.insert(insertAt, newElem)
                        # Accumulate and reset right now
                        foundQuestions.append(currentQuestion)
                        currentQuestion = {}
                    except ValueError:
                        currentQuestion['startIndex'] = i
                        currentQuestion['answerStart'] = text[specEnd + 3:]
                        currentQuestion['startElem'] = elem
                        # Need to accumulate later...
                        break  # no more questions in this string

            if endHere is not None:
                i = endHere + 1  # skip to end of stuff (pre-increment)


def softwalk(root, doc, apply):
    """
    Like panflute's walk method but doesn't replace elements with the
    action's return value, allowing for more flexibility in actions that
    edit the entire document, not just the current node.
    """
    if hasattr(root, 'content'):
        for item in root.content:
            softwalk(item, doc, apply)
    apply(root, doc)


def styleElement():
    """
    Assembles the style tag Panflute element from the `DISCOVERY_CSS`.
    """
    return pf.RawBlock(
        (
            '<style>'
          + DISCOVERY_CSS
          + '</style>'
        ),
        format='html'
    )


def scriptElement():
    """
    Assembles the script tag Panflute element from the `DISCOVERY_JS`.
    """
    return pf.RawBlock(
        (
            '<script type="text/javascript">'
          + DISCOVERY_JS
          + '</script>'
        ),
        format='html'
    )


def main(doc=None):
    """
    Main function just runs `panflute.run_filter`.

    For example:

    >>> import panflute as pf
    >>> t = pf.convert_text(
    ...     'Top\\n\\n[_...][|Short\\nanswer.|]\\n\\nBottom',
    ...     standalone=True
    ... )
    >>> result = main(t)
    >>> expQuestion = pf.Span(
    ...     pf.RawInline('<input type="text">', format='html'),
    ...     pf.RawInline('</input>', format='html'),
    ...     pf.Span(
    ...         pf.Span(
    ...             pf.Str('Short'),
    ...             pf.SoftBreak(),
    ...             pf.Str('answer.')
    ...         ),
    ...         classes=['answer']
    ...     ),
    ...     classes=['question'],
    ...     attributes={'data-flags': ''}
    ... )
    >>> result == pf.Doc(
    ...     styleElement(),
    ...     scriptElement(),
    ...     pf.Para(pf.Str('Top')),
    ...     pf.Para(
    ...         pf.Str(''),
    ...         expQuestion,
    ...         pf.Str('')
    ...     ),
    ...     pf.Para(pf.Str('Bottom'))
    ... )
    True
    """
    printIt = False
    if doc is None:
        doc = pf.load(sys.stdin)
        printIt = True
    softwalk(doc, doc, action)
    doc.content.insert(0, scriptElement())
    doc.content.insert(0, styleElement())
    if printIt:
        pf.dump(doc, sys.stdout)
    return doc


if __name__ == "__main__":
    main()
