#!/usr/bin/env python
"""
A panflute filter for Pandoc JSON which looks for lists of learning goals and
adds interactive elements for students to self-evaluate their progress.
"""

import panflute as pf  # type: ignore

ADD_SELECTS = False
"""
Whether or not to add select drop-downs or not. If set to False, this
filter doesn't make any changes.
"""

LEARNING_GOALS_HEADER_LEVEL = None
"""
Global variable for tracking whether we're in the learning goals
section, and what level header it is.
"""

# HTML for our select element
LG_SELECT = """
<span class="mastery">
(<label>Your mastery:
<select>
    <option>0 - No knowledge</option>
    <option>1 - Learning</option>
    <option>2 - Mastering</option>
    <option>3 - Mastered</option>
</select>
</label>
)
</span>
"""


def action(elem, doc):
    """
    `panflute` action function will be applied to each element in the
    document.
    """
    global LEARNING_GOALS_HEADER_LEVEL

    # Track whether we're in the learning goals section
    if isinstance(elem, pf.Header):
        if (
            LEARNING_GOALS_HEADER_LEVEL is not None
        and elem.level <= LEARNING_GOALS_HEADER_LEVEL
        ):
            LEARNING_GOALS_HEADER_LEVEL = None
        elif (
            LEARNING_GOALS_HEADER_LEVEL is None
        and pf.stringify(elem).strip() == "Learning Goals"
        ):
            LEARNING_GOALS_HEADER_LEVEL = elem.level

    if (
        ADD_SELECTS
    and LEARNING_GOALS_HEADER_LEVEL is not None
    and isinstance(elem, pf.ListItem)
    and not isinstance(
        elem.parent.parent.parent,  # Risky in general; fix this TODO
        (pf.BulletList, pf.OrderedList))
    ):
        elem.content.insert(
            1,
            pf.Plain(
                pf.RawInline(LG_SELECT, format="html"),
            )
        )


def main(doc=None):
    """
    Main function just runs `panflute.run_filter`.
    """
    return pf.run_filter(action, doc=doc)


if __name__ == "__main__":
    main()
