#!/usr/bin/env python
"""
A panflute filter for Pandoc JSON which looks for tables that contains
columns where the header content is just a single vertical bar. These
columns are deleted, but a style attribute is added to the previous
column which adds a right border to that column.
"""

import panflute as pf  # type: ignore

RULES = {
    "_ color _": "background: {}",
    "_ weight _": "font-weight: {}",
    "_ custom _": "{}",
}

BUILTINS = {
    "LG": (
        "background: repeating-linear-gradient(45deg,"
        " #f0f0f0, #f0f0f0 6pt,"
        " #e0e0e0 6pt, #e0e0e0 12pt"
        ");"
    ),
    "LB": (
        "background: repeating-linear-gradient(60deg,"
        " #e0e8ff, #e0e8ff 10pt,"
        " #d0d8ff 10pt, #d0d8ff 20pt"
        ");"
    ),
    "VLB": (
        "background: repeating-linear-gradient(60deg,"
        " #f0f8ff, #f0f8ff 10pt,"
        " #e0e8ff 10pt, #e0e8ff 20pt"
        ");"
    ),
    "LR": (
        "background: repeating-linear-gradient(120deg,"
        " #fee, #fee 10pt,"
        " #fdd 10pt, #fdd 20pt"
        ");"
    ),
    "VLR": (
        "background: repeating-linear-gradient(120deg,"
        " #fff0f0, #fff0f0 10pt,"
        " #ffe0e0 10pt, #ffe0e0 20pt"
        ");"
    ),
    "LY": (
        "background: repeating-linear-gradient(150deg,"
        " #ffd, #ffd 10pt,"
        " #f0f0c8 10pt, #f0f0c8 20pt"
        ");"
    ),
    "VLY": (
        "background: repeating-linear-gradient(150deg,"
        " #fffff0, #fffff0 10pt,"
        " #f0f0e0 10pt, #f0f0e0 20pt"
        ");"
    )
}


def rewriteRow(row, ruleIndexMap):
    """
    Rerwites a `panflute.TableRow` to remove columns at the specified
    indices, while adding styling to the columns after them (does nothing
    except remove the column if used in the last column). If multiple
    columns in a row are included in the indices map, all of their
    styles values will be concatenated.

    The `ruleIndexMap` is a dictionary with row indices as keys, and
    values from the `RULES` dictionary as values (or the value
    `'_ style _'` to use a style from the `BUILTINS`, or the value
    `'_ class _'` to apply a CSS class).

    The contents of each cell in a style column (minus any starting
    `'_ ... _'` value) will be formatted-into the `RULES` value for the
    column's rule (or will be used as a lookup into the `BUILTINS`). The
    resulting style will be applied to the next cell.
    """
    # Iterate in reverse order so column indices remain valid as columns
    # are removed.
    children = row.content
    for index in sorted(ruleIndexMap, reverse=True):
        # Pop column & extract style info
        styleValue = pf.stringify(children.pop(index)).strip()
        # If it starts with '_', get rid of everything until the second
        # '_' (so we can style header rows that have to use _ to declare
        # they're a style column).
        if styleValue.startswith('_'):
            styleValue = styleValue[styleValue.index('_', 1) + 1:].lstrip()

        # Blank value = don't style
        if styleValue == '':
            continue

        # Convert style value according to specified rule
        rule = ruleIndexMap[index]
        applyClass = None
        if rule == '_ class _':
            applyClass = styleValue
        if rule == '_ style _':
            useStyle = BUILTINS.get(styleValue, "")
        else:
            useStyle = RULES.get(rule, '{}').format(styleValue)

        if index < len(children):  # skip if last column
            if applyClass is not None:
                children[index].classes.append(applyClass)
            else:
                attrs = children[index].attributes
                attrs["style"] = attrs.get("style", "") + useStyle


def rewriteCellColorColumns(table):
    """
    Given a `panflute.Table` object, looks for columns where the first
    row of the header starts with '_ style _' (possibly plus whitespace
    and/or a style value afterwards). Rewrites the table by removing each
    cell in such columns, and adding the specified style rules to the
    next cell (see `BUILTINS` for the available built-in styles). Rows
    with empty style cells don't add any style to the next cell.

    The `RULES` dictionary contains alternate directives (besides
    '_ style _') which can be used in header cells to more succinctly
    specify specific custom styles. So for example, if you use
    '_ color _' in your header, each value is interpreted as a
    background color.

    Note that the built-in styles use both color and pattern to give more
    robust visual cues across a range of viewing conditions.

    '_ class _' can also be used to apply a CSS class to cells.
    TODO: '_ class _' isn't working right now :(
    """
    head = table.head
    if len(head.content) == 0:
        return  # no header rows
    firstRow = head.content[0]

    # Make a list of indices for each vertical rule
    styleCols = {}
    for i in range(len(firstRow.content)):
        cell = firstRow.content[i]
        cellStr = pf.stringify(cell).strip()
        if any(
            cellStr.startswith(x)
            for x in ['_ style _', '_ class _'] + list(RULES)
        ):
            styleCols[i] = cellStr[:cellStr.index('_', 1) + 1]

    for hr in head.content:
        rewriteRow(hr, styleCols)

    for body in table.content:
        for row in body.content:
            rewriteRow(row, styleCols)

    # Clean up colspec and colwidth lists
    # (iterate in reverse order so that indices don't break)
    for i in sorted(styleCols, reverse=True):
        table.colspec.pop(i)
        if hasattr(table, "colwidth"):
            table.colwidth.pop(i)


def action(elem, doc):
    """
    `panflute` action function will be applied to each element in the
    document.
    """
    if isinstance(elem, pf.Table):
        rewriteCellColorColumns(elem)


def main(doc=None):
    """
    Main function just runs `panflute.run_filter`.
    """
    return pf.run_filter(action, doc=doc)


if __name__ == "__main__":
    main()
