#!/usr/bin/env python
"""
A panflute filter for Pandoc JSON which looks for tables that contains
columns where the header content is just a single vertical bar. These
columns are deleted, but a style attribute is added to the previous
column which adds a right border to that column.
"""

import panflute as pf

BORDER = "2pt solid black"


def rewriteRow(row, ruleIndices):
    """
    Rerwites a `panflute.TableRow` to remove columns at the specified
    indices, while adding styling to the columns before them (or after if
    column 0 is removed) to create a vertical rule in the table. If
    multiple columns in a row are included in the indices list, Only a
    single border will be added at that point.
    """
    if len(ruleIndices) == 0:
        # Base case: nothing left to remove
        return
    else:
        # Recursive case: remove last column & recurse
        last = ruleIndices[-1]
        if last == 0:
            row.content.pop(0)
            attrs = row.content[0].attributes
            attrs["style"] = attrs.get("style", "") + f"border-left: {BORDER}"
        else:
            row.content.pop(last)
            attrs = row.content[last - 1].attributes
            attrs["style"] = attrs.get("style", "") + f"border-right: {BORDER}"
        # Recurse
        rewriteRow(row, ruleIndices[:-1])


def rewriteVerticalRuleColumns(table):
    """
    Given a `panflute.Table` object, looks for columns where the first
    row of the header contains just a `|` (possibly plus whitespace).
    Rewrites the table by removing each cell in such columns, and adding
    style rules to the previous cell (or next cell if there is no
    previous cell) that add a border to create a vertical rule in the
    table.
    """
    head = table.head
    if len(head.content) == 0:
        return  # no header rows
    firstRow = head.content[0]

    # Make a list of indices for each vertical rule
    vrules = []
    for i in range(len(firstRow.content)):
        cell = firstRow.content[i]
        if pf.stringify(cell).strip() == '|':
            vrules.append(i)

    for hr in head.content:
        rewriteRow(hr, vrules)

    for body in table.content:
        for row in body.content:
            rewriteRow(row, vrules)

    # Clean up colspec and colwidth lists
    # (iterate in reverse order so that indices don't break)
    for i in reversed(vrules):
        table.colspec.pop(i)
        if hasattr(table, "colwidth"):
            table.colwidth.pop(i)


def action(elem, doc):
    """
    `panflute` action function will be applied to each element in the
    document.
    """
    if isinstance(elem, pf.Table):
        rewriteVerticalRuleColumns(elem)


def main(doc=None):
    """
    Main function just runs `panflute.run_filter`.
    """
    return pf.run_filter(action, doc=doc)


if __name__ == "__main__":
    main()
